<?php

namespace Drupal\ratatouille\Service;

use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Entity\ContentEntityTypeInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\default_content\Exporter;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Symfony\Component\Serializer\Serializer;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;

/**
 * Handles fetching and processing of content.
 */
class RecipeContentHandler {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The default content exporter.
   *
   * @var \Drupal\default_content\Exporter
   */
  protected $defaultContentExporter;

  /**
   * The entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected $entityRepository;

  /**
   * The serializer.
   *
   * @var \Symfony\Component\Serializer\Serializer
   */
  protected $serializer;

  /**
   * The YAML serializer.
   *
   * @var \Drupal\Component\Serialization\Yaml
   */
  protected $yaml;

  /**
   * The stream wrapper manager.
   *
   * @var \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface
   */
  protected $streamWrapperManager;

  /**
   * The file URL generator.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * RecipeContentHandler constructor.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, ModuleHandlerInterface $module_handler, Exporter $default_content_exporter, EntityRepositoryInterface $entity_repository, Serializer $serializer, Yaml $yaml, StreamWrapperManagerInterface $stream_wrapper_manager, FileUrlGeneratorInterface $file_url_generator) {
    $this->entityTypeManager = $entity_type_manager;
    $this->moduleHandler = $module_handler;
    $this->defaultContentExporter = $default_content_exporter;
    $this->entityRepository = $entity_repository;
    $this->serializer = $serializer;
    $this->yaml = $yaml;
    $this->streamWrapperManager = $stream_wrapper_manager;
    $this->fileUrlGenerator = $file_url_generator;
  }

  /**
   * Checks if the default_content module is enabled.
   */
  public function isDefaultContentEnabled(): bool {
    return $this->moduleHandler->moduleExists('default_content');
  }

  /**
   * Lists all exportable content entity types.
   *
   * @return array
   *   An associative array of [machine_name => Label].
   */
  public function listExportableEntityTypes(): array {
    $entity_types = $this->entityTypeManager->getDefinitions();
    $exportable_types = array_filter($entity_types, fn($et) => $et instanceof ContentEntityTypeInterface && $et->hasKey('uuid'));

    $type_options = [];
    $labels_seen = [];
    foreach ($exportable_types as $id => $type) {
      $label = $type->getLabel();
      // Ensure labels are unique for choice options.
      // Append machine name if label is not unique.
      if (isset($labels_seen[(string) $label])) {
        $label .= ' (' . $id . ')';
      }
      $type_options[$id] = $label;
      $labels_seen[(string) $type->getLabel()] = TRUE;
    }
    return $type_options;
  }

  /**
   * Gets entity titles keyed by UUID for autocompletion.
   *
   * @return array
   *   An associative array of [uuid => label].
   */
  public function getEntityAutocompleteOptions(string $entity_type_id): array {
    $storage = $this->entityTypeManager->getStorage($entity_type_id);
    $query = $storage->getQuery();
    // Exclude user 1 from export.
    if ($entity_type_id === 'user') {
      $query->condition('uid', 1, '<>');
    }
    $entity_ids = $query->accessCheck(TRUE)->execute();
    $entities = $storage->loadMultiple($entity_ids);

    $autocomplete_options = [];
    $labels_seen = [];
    foreach ($entities as $entity) {
      $label = $entity->label();
      // Ensure labels are unique for choice/autocomplete options.
      // Append UUID if label is not unique.
      if (isset($labels_seen[$label])) {
        $label .= ' (' . substr($entity->uuid(), 0, 8) . ')';
      }
      $autocomplete_options[$entity->uuid()] = $label;
      $labels_seen[$entity->label()] = TRUE;
    }
    return $autocomplete_options;
  }

  /**
   * Exports a single saved entity to a YAML string using its UUID.
   *
   * @param string $recipe_name
   *   The machine name of the recipe being generated.
   * @return string|null
   *   The YAML string or null if export fails.
   */
  public function exportEntity(string $entity_type_id, string $uuid, string $recipe_name, bool $maintain_file_structure): ?string {
    $entity = $this->entityRepository->loadEntityByUuid($entity_type_id, $uuid);
    if ($entity) {
      // For any entity that is saved in the database, use the default content
      // exporter service, which is more reliable.
      $yaml_string = $this->defaultContentExporter->exportContent($entity->getEntityTypeId(), $entity->id());
      $yaml_data = $this->yaml->decode($yaml_string);

      // Process the resulting array to fix file URIs.
      $processed_yaml_data = $this->processFileUrisInYaml($yaml_data, $maintain_file_structure);

      return $this->yaml->encode($processed_yaml_data);
    }
    return NULL;
  }

  /**
   * Recursively processes a YAML array to replace file URIs.
   *
   * @param array $data
   *   The YAML data array passed by reference.
   * @param bool $maintain_file_structure
   *   Whether the file structure is maintained in the recipe.
   * @return array
   *   The processed YAML data array.
   */
  private function processFileUrisInYaml(array &$data, bool $maintain_file_structure): array {
    array_walk_recursive($data, function (&$value, $key) use ($maintain_file_structure) {
      if ($key === 'value' && is_string($value) && preg_match('/^(public|private|temporary):\/\//', $value)) {
        $value = $this->rewriteUri($value, $maintain_file_structure);
      }
    });
    return $data;
  }

  /**
   * Rewrites a Drupal stream wrapper URI to a recipe stream wrapper URI.
   */
  private function rewriteUri(string $uri, bool $maintain_file_structure): string {
    $scheme = $this->streamWrapperManager->getScheme($uri);
    if ($scheme && in_array($scheme, ['public', 'private', 'temporary'])) {
      $relative_path = substr($uri, strlen($scheme . '://'));
      if ($maintain_file_structure) {
        return 'recipe://' . $relative_path;
      }
      else {
        $filename = $scheme . '_' . basename($this->fileUrlGenerator->getPathFromUri($uri));
        return 'recipe://' . $filename;
      }
    }
    return $uri;
  }

  /**
   * Rewrites a hardcoded file URL to a recipe file path.
   */
  private function rewriteUrl(string $url, string $recipe_name, bool $maintain_file_structure): string {
    if ($maintain_file_structure) {
      $relative_path_from_files = implode('/', array_slice(explode('/', $url), 4));
      return '/recipes/' . $recipe_name . '/files/' . $relative_path_from_files;
    }
    else {
      $filename = basename($url);
      $scheme_prefix = '';
      if (str_starts_with($url, '/sites/default/files/public/')) {
        $scheme_prefix = 'public_';
      }
      elseif (str_starts_with($url, '/sites/default/files/private/')) {
        $scheme_prefix = 'private_';
      }
      return '/recipes/' . $recipe_name . '/files/' . $scheme_prefix . $filename;
    }
  }

}
