<?php

namespace Drupal\ratatouille\Service;

use Drupal\Core\Config\ConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;
use Drupal\default_content\Exporter;
use Drupal\Component\Serialization\Yaml;
use Psr\Log\LoggerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Resolves all dependencies for a recipe.
 */
class RecipeDependencyResolver {

  /**
   * The config manager.
   *
   * @var \Drupal\Core\Config\ConfigManagerInterface
   */
  protected $configManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected $entityRepository;

  /**
   * The content config dependency collector.
   *
   * @var \Drupal\ratatouille\Service\ContentConfigDependencyCollector
   */
  protected $contentConfigDependencyCollector;

  /**
   * The file URL generator.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * The file system.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The stream wrapper manager.
   *
   * @var \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface
   */
  protected $streamWrapperManager;

  /**
   * The default content exporter.
   *
   * @var \Drupal\default_content\Exporter
   */
  protected $defaultContentExporter;

  /**
   * The YAML serializer.
   *
   * @var \Drupal\Component\Serialization\Yaml
   */
  protected $yaml;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * RecipeDependencyResolver constructor.
   */
  public function __construct(ConfigManagerInterface $config_manager, EntityTypeManagerInterface $entity_type_manager, ModuleHandlerInterface $module_handler, EntityRepositoryInterface $entity_repository, ContentConfigDependencyCollector $content_config_dependency_collector, FileUrlGeneratorInterface $file_url_generator, FileSystemInterface $file_system, StreamWrapperManagerInterface $stream_wrapper_manager, Exporter $default_content_exporter, Yaml $yaml, LoggerChannelFactoryInterface $logger_factory) {
    $this->configManager = $config_manager;
    $this->entityTypeManager = $entity_type_manager;
    $this->moduleHandler = $module_handler;
    $this->entityRepository = $entity_repository;
    $this->contentConfigDependencyCollector = $content_config_dependency_collector;
    $this->fileUrlGenerator = $file_url_generator;
    $this->fileSystem = $file_system;
    $this->streamWrapperManager = $stream_wrapper_manager;
    $this->defaultContentExporter = $default_content_exporter;
    $this->yaml = $yaml;
    $this->logger = $logger_factory->get('ratatouille');
  }

  /**
   * Resolves all dependencies for the selected items.
   *
   * @return array
   *   An array containing three elements: [module_dependencies, config_dependencies, content_dependencies, file_dependencies].
   */
  public function resolve(array $selected_config, array $selected_content): array {
    $module_deps = [];
    $final_config_deps = []; // This will be the final, unique list of configs.
    $content_deps = $selected_content; // Start with the user-selected content.
    $file_deps = []; // This will be an array of file URIs.

    // 1. Start with the user's selected config as the initial list to scan.
    $configs_to_scan = $selected_config;

    // 2. Find config dependencies from content and add them to the scan list.
    if (!empty($content_deps)) {
      $this->resolveContentDependencies($content_deps, $configs_to_scan, $module_deps, $file_deps);
    }

    // Make sure the scan list is unique before processing.
    $configs_to_scan = array_unique($configs_to_scan);

    // 3. Now, recursively process every item in the combined scan list.
    foreach ($configs_to_scan as $config_name) {
      $this->collectConfigDependencies($config_name, $final_config_deps, $module_deps);
    }

    return [array_unique($module_deps), array_unique($final_config_deps), $content_deps, array_unique($file_deps)];
  }

  /**
   * Recursively collects dependencies for a given configuration item.
   */
  private function collectConfigDependencies(string $config_name, array &$collected_config_names, array &$module_dependencies) {
    // If this config has already been processed, skip it.
    if (in_array($config_name, $collected_config_names)) {
      return;
    }
    // Add the current config to the list to prevent infinite loops.
    $collected_config_names[] = $config_name;

    $config = $this->configManager->getConfigFactory()->get($config_name);
    $all_deps = $config->get('dependencies') ?: [];

    if (!empty($all_deps['config'])) {
      foreach ($all_deps['config'] as $dep_name) {
        $this->collectConfigDependencies($dep_name, $collected_config_names, $module_dependencies);
      }
    }

    if (!empty($all_deps['module'])) {
      foreach ($all_deps['module'] as $module) {
        $module_dependencies['drupal:' . $module] = 'drupal:' . $module;
      }
    }

    // Also add the module that provides this configuration.
    $config_prefix = explode('.', $config_name)[0];
    if ($this->moduleHandler->moduleExists($config_prefix)) {
      $module_dependencies['drupal:' . $config_prefix] = 'drupal:' . $config_prefix;
    }

    // Special handling for content type configurations to include their default form/view displays.
    if (str_starts_with($config_name, 'node.type.') || str_starts_with($config_name, 'media.type.') || str_starts_with($config_name, 'block_content.type.') || str_starts_with($config_name, 'taxonomy.vocabulary.')) {
      $parts = explode('.', $config_name);
      $entity_type_id = $parts[0]; // e.g., node, media
      $bundle_id = $parts[2]; // e.g., article, image

      // Add form display config.
      $form_display_config_name = 'core.entity_form_display.' . $entity_type_id . '.' . $bundle_id . '.default';
      if ($this->configManager->getConfigFactory()->get($form_display_config_name)->getRawData()) {
        if (!in_array($form_display_config_name, $collected_config_names)) {
          $collected_config_names[] = $form_display_config_name;
          // Recursively collect dependencies for the newly added display config.
          $this->collectConfigDependencies($form_display_config_name, $collected_config_names, $module_dependencies);
        }
      }

      // Add view display config.
      $view_display_config_name = 'core.entity_view_display.' . $entity_type_id . '.' . $bundle_id . '.default';
      if ($this->configManager->getConfigFactory()->get($view_display_config_name)->getRawData()) {
        if (!in_array($view_display_config_name, $collected_config_names)) {
          $collected_config_names[] = $view_display_config_name;
          // Recursively collect dependencies for the newly added display config.
          $this->collectConfigDependencies($view_display_config_name, $collected_config_names, $module_dependencies);
        }
      }
    }
  }

  /**
   * Resolves dependencies for content entities.
   *
   * @param array $file_deps
   *   The array of file URIs to collect.
   */
  private function resolveContentDependencies(array &$content_deps, array &$config_deps, array &$module_deps, array &$file_deps) {
    $module_deps['drupal:default_content'] = 'drupal:default_content';

    $processed_uuids = [];

    // Use a while loop to keep processing until no new dependencies are found.
    $found_new_dependencies = TRUE;
    while ($found_new_dependencies) {
      $found_new_dependencies = FALSE;
      foreach ($content_deps as $entity_type_id => $uuids) {
        foreach ($uuids as $uuid) {
          if (isset($processed_uuids[$uuid])) {
            continue;
          }

          $entity = $this->entityRepository->loadEntityByUuid($entity_type_id, $uuid);
          if ($entity) {
            $this->collectReferencedEntitiesAndFiles($entity, $content_deps, $module_deps, $config_deps, $file_deps, $found_new_dependencies);
          }
          $processed_uuids[$uuid] = TRUE;
        }
      }
    }
  }

  /**
   * Recursively collects referenced entities and their files.
   */
  private function collectReferencedEntitiesAndFiles($entity, array &$content_deps, array &$module_deps, array &$config_deps, array &$file_deps, bool &$found_new_dependencies) {
    // Add the current entity's bundle and field config dependencies.
    $this->contentConfigDependencyCollector->addBundleConfig($entity->getEntityTypeId(), $entity->bundle(), $config_deps);
    $this->contentConfigDependencyCollector->addFieldConfig($entity->getEntityTypeId(), $entity->bundle(), $config_deps);

    // Add the module that provides this entity type.
    $provider = $entity->getEntityType()->getProvider();
    if ($provider) {
      $module_deps['drupal:' . $provider] = 'drupal:' . $provider;
    }

    // Process _meta.depends from default_content exports.
    if ($entity->id() !== NULL) {
      $exported_yaml_string = $this->defaultContentExporter->exportContent($entity->getEntityTypeId(), $entity->id());
      if ($exported_yaml_string) {
        $exported_data = $this->yaml->decode($exported_yaml_string);
        if (isset($exported_data['_meta']['depends']) && is_array($exported_data['_meta']['depends'])) {
          foreach ($exported_data['_meta']['depends'] as $dep_uuid => $dep_entity_type_id) {
            if (!empty($dep_entity_type_id) && !empty($dep_uuid)) {
              if (!isset($content_deps[$dep_entity_type_id]) || !in_array($dep_uuid, $content_deps[$dep_entity_type_id])) {
                $content_deps[$dep_entity_type_id][] = $dep_uuid;
                $found_new_dependencies = TRUE;
              }
            }
          }
        }
      }
    }

    // Process entity reference fields.
    foreach ($entity->getFields() as $field) {
      if (!$field->getFieldDefinition()->isInternal() && !$field->isEmpty()) {
        if ($field->getFieldDefinition()->getType() === 'entity_reference' || $field->getFieldDefinition()->getType() === 'entity_reference_revisions') {
          foreach ($field->referencedEntities() as $referenced_entity) {
            // Skip user 1 from auto-discovery.
            if ($referenced_entity->getEntityTypeId() === 'user' && $referenced_entity->id() === '1') {
              continue;
            }
            if ($referenced_entity->getEntityType()->getBundleEntityType()) {
              $ref_uuid = $referenced_entity->uuid();
              $ref_type = $referenced_entity->getEntityTypeId();
              if (!isset($content_deps[$ref_type]) || !in_array($ref_uuid, $content_deps[$ref_type])) {
                $content_deps[$ref_type][] = $ref_uuid;
                $found_new_dependencies = TRUE;
              }
            }
          }
        }
        // Handle file/image/media fields to collect file URIs.
        elseif (in_array($field->getFieldDefinition()->getType(), ['file', 'image', 'media'])) {
          foreach ($field->referencedEntities() as $referenced_entity) {
            if ($referenced_entity->getEntityTypeId() === 'file') {
              $uri = $referenced_entity->getFileUri();
              if ($uri && !in_array($uri, $file_deps) && $this->streamWrapperManager->getScheme($uri) !== 'http' && $this->streamWrapperManager->getScheme($uri) !== 'https') {
                $file_deps[] = $uri;
              }
            }
            elseif ($referenced_entity->getEntityTypeId() === 'media') {
              $source_field_name = $referenced_entity->getSource()->getSourceFieldDefinition($referenced_entity->bundle())->getName();
              if ($referenced_entity->hasField($source_field_name)) {
                foreach ($referenced_entity->get($source_field_name)->referencedEntities() as $file_entity) {
                  if ($file_entity->getEntityTypeId() === 'file') {
                    $uri = $file_entity->getFileUri();
                    if ($uri && !in_array($uri, $file_deps) && $this->streamWrapperManager->getScheme($uri) !== 'http' && $this->streamWrapperManager->getScheme($uri) !== 'https') {
                      $file_deps[] = $uri;
                    }
                  }
                }
              }
            }
          }
        }
      }
    }

    // Ensure content_translation module is added if it's enabled on the source site.
    // This is a pragmatic fix for base fields like 'content_translation_source'
    // which do not have explicit config dependencies.
    if ($this->moduleHandler->moduleExists('content_translation')) {
      $module_deps['drupal:content_translation'] = 'drupal:content_translation';
    }
  }

  /**
   * Resolves configuration dependencies for a given set of config names.
   *
   * @param array $config_names
   *   An array of configuration names to resolve dependencies for.
   *
   * @return array
   *   An array containing two elements: [config_dependencies, module_dependencies].
   */
  public function resolveConfigDependencies(array $config_names): array {
    $collected_config_deps = [];
    $collected_module_deps = [];

    foreach ($config_names as $config_name) {
      $this->collectConfigDependencies($config_name, $collected_config_deps, $collected_module_deps);
    }

    return [array_unique($collected_config_deps), array_unique($collected_module_deps)];
  }
}