<?php

namespace Drupal\ratatouille\Service;

use Drupal\Component\Serialization\Yaml;
use Drupal\ratatouille\Exception\RecipeGenerationException;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;

/**
 * Handles the generation of recipe files.
 */
class RecipeFileGenerator {

  /**
   * The YAML serialization service.
   *
   * @var \Drupal\Component\Serialization\Yaml
   */
  protected $yaml;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The stream wrapper manager service.
   *
   * @var \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface
   */
  protected $streamWrapperManager;

  /**
   * RecipeFileGenerator constructor.
   *
   * @param \Drupal\Component\Serialization\Yaml $yaml
   *   The YAML serialization service.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface $stream_wrapper_manager
   *   The stream wrapper manager service.
   */
  public function __construct(Yaml $yaml, FileSystemInterface $file_system, StreamWrapperManagerInterface $stream_wrapper_manager) {
    $this->yaml = $yaml;
    $this->fileSystem = $file_system;
    $this->streamWrapperManager = $stream_wrapper_manager;
  }

  /**
   * Creates the recipe directory structure.
   *
   * @param string $name
   *   The machine name of the recipe.
   *
   * @return string|false
   *   The path to the recipe directory or FALSE on failure.
   */
  public function createRecipeDirectory(string $name): string|false {
    $path = DRUPAL_ROOT . '/recipes/custom/' . $name;
    if (is_dir($path)) {
      // The decision to overwrite should be handled by the calling command.
      // For this service, we just ensure the directories exist.
    }
    if (!@mkdir($path . '/config', 0755, TRUE) || !@mkdir($path . '/content', 0755, TRUE) || !@mkdir($path . '/files', 0755, TRUE)) {
      throw new RecipeGenerationException(sprintf('Failed to create recipe directory at %s', $path));
    }
    return $path;
  }

  /**
   * Writes the main recipe.yml file.
   */
  public function writeRecipeYaml(string $recipe_path, array $recipe_data) {
    // Remove content key if it exists and is empty.
    if (isset($recipe_data['content']) && empty($recipe_data['content'])) {
      unset($recipe_data['content']);
    }

    file_put_contents($recipe_path . '/recipe.yml', $this->yaml->encode($recipe_data));
  }

  /**
   * Writes the composer.json file.
   */
  public function writeComposerJson(string $recipe_path, string $name, array $composer_dependencies) {
    if (empty($composer_dependencies)) {
      return;
    }
    $composer_data = ['name' => 'drupal/' . $name, 'type' => 'drupal-recipe', 'require' => []];
    foreach ($composer_dependencies as $dep) {
      if (strpos($dep, ':') !== false) {
        list($package, $version) = explode(':', $dep, 2);
        $composer_data['require'][$package] = $version;
      }
    }
    file_put_contents($recipe_path . '/composer.json', json_encode($composer_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
  }

  /**
   * Writes all configuration files.
   *
   * @return int
   *   The number of files written.
   */
  public function writeConfigurationFiles(string $recipe_path, array $all_configs_to_write): int {
    $config_path = $recipe_path . '/config';
    foreach ($all_configs_to_write as $config_name => $config_data) {
      if (isset($config_data['uuid'])) {
        unset($config_data['uuid']);
      }
      // Core settings are not prefixed and can cause issues.
      if (strpos($config_name, '.') === false) {
          $config_name = $config_name . '.settings';
      }
      $file_path = $config_path . '/' . $config_name . '.yml';
      file_put_contents($file_path, $this->yaml->encode($config_data));
    }
    return count($all_configs_to_write);
  }

  /**
   * Writes a single content entity file.
   */
  public function writeContentFile(string $recipe_path, string $entity_type_id, string $uuid, string $exported_yaml): string|false {
    $entity_type_dir = $recipe_path . '/content/' . $entity_type_id;
    @mkdir($entity_type_dir, 0755, TRUE);
    $file_name = $uuid . '.yml';
    $full_file_path = $entity_type_dir . '/' . $file_name;
    if (file_put_contents($full_file_path, $exported_yaml)) {
      return $entity_type_id . '/' . $file_name;
    }
    return FALSE;
  }

  /**
   * Copies files to the recipe's files directory.
   *
   * @param string $recipe_path
   *   The base path of the recipe.
   * @param array $file_uris
   *   An array of file URIs to copy.
   * @param bool $maintain_structure
   *   Whether to maintain the original folder structure within the recipe's files directory.
   * @return array
   *   An array of destination paths for the copied files.
   */
  public function copyFiles(string $recipe_path, array $file_uris, bool $maintain_structure): array {
    $copied_files = [];
    $destination_base = $recipe_path . '/files';

    foreach ($file_uris as $uri) {
      try {
        $stream_wrapper_name = $this->streamWrapperManager->getScheme($uri);
        $source_path = $this->fileSystem->realpath($uri);
        
        // Log source path for debugging.
        \Drupal::logger('ratatouille')->debug('Attempting to copy file from source: @source_path', ['@source_path' => $source_path]);

        $relative_path = '';
        if ($maintain_structure) {
          // Get the path relative to the stream wrapper's root.
          $stream_wrapper_uri = $stream_wrapper_name . '://';
          if (str_starts_with($uri, $stream_wrapper_uri)) {
            $relative_path = substr($uri, strlen($stream_wrapper_uri));
          }
          $destination_uri = $destination_base . '/' . $stream_wrapper_name . '/' . $relative_path;
        } else {
          // Flat structure: filename prefixed with stream wrapper.
          $filename = $stream_wrapper_name . '_' . basename($source_path);
          $destination_uri = $destination_base . '/' . $filename;
        }

        // Log destination path for debugging.
        \Drupal::logger('ratatouille')->debug('Attempting to copy file to destination: @destination_uri', ['@destination_uri' => $destination_uri]);

        // Ensure the destination directory exists.
        $this->fileSystem->prepareDirectory(dirname($destination_uri), FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);

        // Copy the file.
        if ($this->fileSystem->copy($source_path, $destination_uri, FileSystemInterface::EXISTS_REPLACE)) {
          $copied_files[] = $destination_uri;
          \Drupal::logger('ratatouille')->debug('Successfully copied file to: @destination_uri', ['@destination_uri' => $destination_uri]);
        } else {
          \Drupal::logger('ratatouille')->error('Failed to copy file from @source_path to @destination_uri', ['@source_path' => $source_path, '@destination_uri' => $destination_uri]);
        }
      }
      catch (\Exception $e) {
        \Drupal::logger('ratatouille')->error('Failed to copy file @uri: @message', [
          '@uri' => $uri,
          '@message'=> $e->getMessage(),
        ]);
      }
    }
    return $copied_files;
  }

}