<?php

namespace Drupal\raven\Config;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ConfigFactoryOverrideInterface;
use Drupal\Core\Config\StorageInterface;
use Drupal\Core\Url;
use Sentry\Dsn;

/**
 * Automatically overrides Security Kit configuration.
 */
class SecKitOverrides implements ConfigFactoryOverrideInterface {

  /**
   * Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Sentry environment.
   *
   * @var ?string
   */
  protected $environment;

  /**
   * Constructs the Security Kit config overrider.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory object.
   * @param ?string $environment
   *   Sentry environment.
   */
  public function __construct(ConfigFactoryInterface $config_factory, ?string $environment = NULL) {
    $this->configFactory = $config_factory;
    $this->environment = $environment;
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore-next-line Core doesn't yet document the param or return arrays.
   */
  public function loadOverrides($names) {
    $overrides = [];
    if (!in_array('seckit.settings', $names)) {
      return $overrides;
    }
    if (!class_exists(Dsn::class)) {
      return $overrides;
    }
    $config = $this->configFactory->get('raven.settings');
    $dsn = empty($_SERVER['SENTRY_DSN']) ? $config->get('public_dsn') : $_SERVER['SENTRY_DSN'];
    if (NULL === $dsn) {
      return $overrides;
    }
    try {
      $dsn = Dsn::createFromString($dsn);
    }
    catch (\InvalidArgumentException $e) {
      // Raven is incorrectly configured.
      return $overrides;
    }
    if ($config->get('seckit_set_report_uri')) {
      $query['sentry_environment'] = empty($_SERVER['SENTRY_ENVIRONMENT']) ? ($config->get('environment') ?: $this->environment) : $_SERVER['SENTRY_ENVIRONMENT'];
      if ($release = empty($_SERVER['SENTRY_RELEASE']) ? $config->get('release') : $_SERVER['SENTRY_RELEASE']) {
        $query['sentry_release'] = $release;
      }
      $overrides['seckit.settings']['seckit_xss']['csp']['report-uri'] = $overrides['seckit.settings']['seckit_ct']['report_uri'] =
        Url::fromUri($dsn->getCspReportEndpointUrl(), ['query' => $query])->toString();
    }
    if ($config->get('javascript_error_handler')) {
      $seckitConfig = $this->configFactory->getEditable('seckit.settings');
      if ($config->get('show_report_dialog')) {
        $src[] = str_replace(
          ["/{$dsn->getProjectId(TRUE)}/", '/store/'],
          ['/embed/', '/error-page/'],
          $dsn->getStoreApiEndpointUrl()
        );
        if ($url = $config->get('error_embed_url')) {
          $src[] = "$url/api/embed/error-page/";
        }
        if ($script_src = $seckitConfig->get('seckit_xss.csp.script-src') ?: $seckitConfig->get('seckit_xss.csp.default-src')) {
          $overrides['seckit.settings']['seckit_xss']['csp']['script-src'] = implode(' ', array_merge([$script_src], $src));
        }
        if ($img_src = $seckitConfig->get('seckit_xss.csp.img-src') ?: $seckitConfig->get('seckit_xss.csp.default-src')) {
          $img = explode(' ', $img_src);
          $img[] = 'data:';
          $overrides['seckit.settings']['seckit_xss']['csp']['img-src'] = implode(' ', array_unique($img));
        }
        if ($style_src = $seckitConfig->get('seckit_xss.csp.style-src') ?: $seckitConfig->get('seckit_xss.csp.default-src')) {
          $style = explode(' ', $style_src);
          $style[] = "'unsafe-inline'";
          $overrides['seckit.settings']['seckit_xss']['csp']['style-src'] = implode(' ', array_unique($style));
        }
      }
      if ($connect_src = $seckitConfig->get('seckit_xss.csp.connect-src') ?: $seckitConfig->get('seckit_xss.csp.default-src')) {
        $connect = [$connect_src];
        if (!$config->get('tunnel')) {
          $connect[] = $dsn->getStoreApiEndpointUrl();
          $connect[] = $dsn->getEnvelopeApiEndpointUrl();
        }
        if (isset($src)) {
          $connect = array_merge($connect, $src);
        }
        $overrides['seckit.settings']['seckit_xss']['csp']['connect-src'] = implode(' ', $connect);
      }
    }
    return $overrides;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheSuffix() {
    return 'RavenSecKitOverrider';
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheableMetadata($name) {
    return new CacheableMetadata();
  }

  /**
   * Creates a configuration object for use during install and synchronization.
   *
   * @param string $name
   *   The configuration object name.
   * @param string $collection
   *   The configuration collection.
   *
   * @return \Drupal\Core\Config\StorableConfigBase|null
   *   The configuration object for the provided name and collection.
   */
  public function createConfigObject($name, $collection = StorageInterface::DEFAULT_COLLECTION) {
    return NULL;
  }

}
