<?php

namespace Drupal\raven\Plugin\CspReportingHandler;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Url;
use Drupal\csp\Csp;
use Drupal\csp\Plugin\ReportingHandlerBase;
use Drupal\raven\Config\ConfigInterface;
use Sentry\Dsn;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * CSP Reporting Plugin for a Sentry endpoint.
 *
 * @CspReportingHandler(
 *   id = "raven",
 *   label = "Sentry",
 *   description = @Translation("Reports will be sent to Sentry."),
 * )
 */
class Raven extends ReportingHandlerBase implements ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore missingType.iterableValue
   */
  final public function __construct(array $configuration, $plugin_id, $plugin_definition, protected ConfigInterface $config) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore missingType.iterableValue
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      // @phpstan-ignore argument.type
      $plugin_definition,
      $container->get('raven.config'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function alterPolicy(Csp $policy): void {
    $dsn = $this->config->getFrontendDsn();
    if (NULL === $dsn) {
      return;
    }
    try {
      $dsn = Dsn::createFromString($dsn);
    }
    catch (\InvalidArgumentException $e) {
      // Raven is incorrectly configured.
      return;
    }
    $query['sentry_environment'] = $this->config->getEnvironment();
    if ($release = $this->config->getRelease()) {
      $query['sentry_release'] = $release;
    }
    $policy->setDirective('report-uri', Url::fromUri($dsn->getCspReportEndpointUrl(), ['query' => $query])->toString());
  }

}
