/**
 * @file
 * JS handler for reading progress bar.
 */
((Drupal, once) => {
  function isElementVisible(el) {
    const style = window.getComputedStyle(el);
    return style.display !== 'none';
  }
  function emitUpdateEvent(progressBar) {
    const percentage =
      progressBar.max > 0 ? (progressBar.value / progressBar.max) * 100 : 0;
    const event = new CustomEvent('readingProgressBarUpdate', {
      detail: {
        percentage: Math.round(percentage),
        progressBar, // Include reference to the emitting element.
      },
    });
    document.dispatchEvent(event);
  }
  function updateProgressBarValue(progressBar) {
    const value =
      progressBar._rpbContainer === document.documentElement
        ? window.scrollY
        : -Math.round(progressBar._rpbContainer.getBoundingClientRect().top);
    progressBar.value = Math.min(Math.max(value, 0), progressBar.max);
    emitUpdateEvent(progressBar);
  }
  function setupProgressBar(progressBar) {
    /* Set the max scrollable area */
    const viewportHeight = window.innerHeight;
    const containerHeight = progressBar._rpbContainer.scrollHeight;
    progressBar.max = containerHeight - viewportHeight;
    const minimumContainerViewportRatio = progressBar.dataset.minRatio || 2;
    if (containerHeight / viewportHeight > minimumContainerViewportRatio) {
      progressBar.style.display = '';
    } else {
      progressBar.style.display = 'none';
    }
  }
  function initProgressBar(progressBar) {
    const autoHideDelay = progressBar.dataset.hideDelay || 0;
    // Define the container to be used with the reading progress bar.
    progressBar._rpbContainer = progressBar.dataset.containerSelector
      ? document.querySelector(progressBar.dataset.containerSelector) ||
        document.documentElement
      : document.documentElement;
    // Initial setup.
    setupProgressBar(progressBar);
    // Refresh setup when the page is fully loaded.
    window.addEventListener('load', () => {
      setupProgressBar(progressBar);
    });
    // Refresh setup when the viewport is resized.
    window.addEventListener('resize', () => {
      setupProgressBar(progressBar);
    });
    let autoHideTimer = null;
    document.addEventListener('scroll', () => {
      // Update progress bar if visible only.
      if (isElementVisible(progressBar)) {
        updateProgressBarValue(progressBar);
        if (progressBar.classList.contains('hidden')) {
          progressBar.classList.remove('hidden');
        }
        if (autoHideDelay > 0) {
          if (autoHideTimer) {
            clearTimeout(autoHideTimer);
          }
          autoHideTimer = setTimeout(() => {
            progressBar.classList.add('hidden');
          }, autoHideDelay);
        }
      }
    });
    // Refresh progressbar when the content container is resized.
    const resizeObserver = new ResizeObserver((entries) => {
      entries.forEach((entry) => {
        if (entry.target === progressBar._rpbContainer) {
          setupProgressBar(progressBar);
        }
      });
    });
    resizeObserver.observe(progressBar._rpbContainer);
  }

  Drupal.behaviors.reading_progress_bar = {
    attach(context) {
      const progressBars = once('reading_progress_bar', 'progress', context);
      progressBars.forEach((progressBar) => {
        initProgressBar(progressBar);
      });
    },
  };
})(Drupal, once);
