(function drupal(Drupal) {
  Drupal.behaviors.readingRating = {
    attach: function attach(context, settings) {
      once('readingrating', '.reading-rating').forEach(function (element) {

        /**
         * Wait for CKEditor5 and get ID, if not, this is a standard text
         * field where we only need the value.
         */
        setTimeout(function () {
          const ckEditorId = element.dataset.ckeditor5Id;
          if (ckEditorId) {
            rrCkeditor5(Drupal.CKEditor5Instances.get(ckEditorId));
          } else {
            rrUpdateStats(element, element.value);
            ['keyup', 'change', 'paste'].forEach(function (evt) {
              element.addEventListener(evt, function () {
                rrUpdateStats(element, element.value);
              });
            });
          }
        }, 10);
      });

      // Strips HTML and newline tags.
      function stripTags(input) {
        const normalizedText = input
          .replace(/(<([^>]+)>)/gi, ' ') // Replace HTML tags
          .replace(/(\r\n|\n|\r)/gm, ' ') // Replace new lines (in many forms)
          .replace(/^\s+|\s+$/g, ' ') // Replace leading or trailing multiple spaces
          .replace('&nbsp;', ' '); // Replaces most non-breaking-spaces

        return normalizedText;
      }

      function rrUpdateStats(textArea, text) {
        if (typeof textstatistics === 'undefined') {
          console.warn(
            "The text-statistics.js library did not load properly or doesn't exist.",
          );
          return;
        }

        let target = null;
        const readingRatingContainer =
          document.querySelector(`div.rr-container[data-textarea="${textArea.id}"]`);

        const ratingListLabels = Array.from(
          readingRatingContainer.querySelectorAll('[data-rr-option]'),
        );
        const ratingListLabelsGrades = Array.from(
          readingRatingContainer.querySelectorAll('[data-rr-option-grade]'),
        );

        const stripped = stripTags(text);
        const ratingOptions = getRatingOptions(stripped);

        targetGeneral = readingRatingContainer.querySelector(
          '[data-rr-option="' + ratingOptions.general + '"]',
        );

        targetGrade = readingRatingContainer.querySelector(
          '[data-rr-option-grade="' + ratingOptions.grade + '"]',
        );

        // Update ratings.
        if (targetGeneral) {
          ratingListLabels.forEach(function (key) {
            key.setAttribute('aria-current', false);
          });
          targetGeneral.setAttribute('aria-current', true);
        }

        if (targetGrade) {
          ratingListLabelsGrades.forEach(function (key) {
            key.setAttribute('aria-current', false);
          });
          targetGrade.setAttribute('aria-current', true);
        }
      }

      function rrCkeditor5(editor) {
        once('readingratingckeditor', editor.sourceElement).forEach(
          function (ckEditorTextArea) {
            let textData = editor.getData();
            rrUpdateStats(ckEditorTextArea, textData);

            ['keyup', 'change', 'paste'].forEach(function (evt) {
              editor.model.document.on(evt, function () {
                textData = editor.getData();
                rrUpdateStats(ckEditorTextArea, textData);
              });
            });
          },
        );
      }

      function getRatingOptions(text) {
        let ratingOptions = {
          general: 'easy',
          grade: 'grade_very_easy'
        };

        if (!text) {
          ratingOptions = {
            general: 'unknown',
            grade: 'unknown'
          };
        }

        const textStats = new textstatistics();
        if (text) {
          const rating = textStats.fleschKincaidReadingEase(text);
          const ratingGrade = textStats.fleschKincaidGradeLevel(text);

          switch (true) {
            case rating < 50:
              ratingOptions.general = 'difficult';
              break;

            case rating < 60:
              ratingOptions.general = 'moderate';
              break;
          }

          switch (true) {
            case ratingGrade <= 5:
              ratingOptions.grade = 'grade_very_easy';
              break;

            case ratingGrade <= 8:
              ratingOptions.grade = 'grade_easy';
              break;

            case ratingGrade <= 12:
              ratingOptions.grade = 'grade_fairly_easy';
              break;

            case ratingGrade <= 16:
              ratingOptions.grade = 'grade_moderate';
              break;

            case ratingGrade <= 17:
              ratingOptions.grade = 'grade_difficult';
              break;
          }
        }

        return ratingOptions;
      }
    },
  };
})(Drupal);
