<?php

namespace Drupal\reading_rating;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Trusted render callbacks.
 */
class ReadingRatingCallbacks implements TrustedCallbackInterface, ContainerInjectionInterface {

  use StringTranslationTrait;

  /**
   * Configuration Factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $rrConfig;

  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructs a new ReadingRatingCallbacks object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory interface.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    RendererInterface $renderer,
  ) {
    $this->rrConfig = $config_factory->getEditable('reading_rating.settings');
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('renderer'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public static function trustedCallbacks() {
    return ['readingRatingProcess', 'readingRatingPreRender'];
  }

  /**
   * Static pre-render callback that delegates to the service instance.
   *
   * @param array $element
   *   The element to process.
   * @param Drupal\Core\Form\FormStateInterface $form_state
   *   The form state interface.
   * @param array $complete_form
   *   The complete form array.
   *
   * @return array
   *   The processed element.
   */
  public static function readingRatingProcess(array $element, FormStateInterface $form_state, array &$complete_form) {
    return \Drupal::service('reading_rating.renderer')
      ->doReadingRatingProcess($element, $form_state, $complete_form);
  }

  /**
   * Process function to set reading rating attributes.
   *
   * @param array|mixed $element
   *   The render array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param array $complete_form
   *   The complete form structure.
   *
   * @return array
   *   The processed render array.
   */
  protected function doReadingRatingProcess($element, $form_state, $complete_form) {
    if (isset($element['#enable_reading_rating']) && $element['#enable_reading_rating'] === TRUE) {
      $element['#attributes']['class'][] = 'reading-rating';
      $element['#attached']['library'][] = 'reading_rating/reading_rating';

      $options = [];

      if (isset($element['#enable_grade_level']) && $element['#enable_grade_level']) {
        $options['grade'] = TRUE;
      }

      $widget = [
        '#theme' => 'reading_rating_widget',
        '#textarea_id' => $element['#id'],
        '#text_replacements' => $this->rrConfig->get('text_replacements'),
        '#options' => $options,
      ];

      $element['#suffix'] = isset($element['#suffix']) ?
      $element['#suffix'] . $this->renderer->render($widget) :
      $this->renderer->render($widget);

    }
    return $element;
  }

  /**
   * Deprecated PreRender function. Alerts in log to run updb.
   *
   * @param array|mixed $element
   *   The render array.
   *
   * @return array
   *   The processed render array.
   */
  public static function readingRatingPreRender($element) {
    $message = t('Reading Rating module has been updated. Please run database updates to complete the installation. <code>drush updb</code>');
    \Drupal::logger('reading_rating')->warning($message);

    return $element;
  }

}
