<?php

declare(strict_types=1);

namespace Drupal\rebuilder\Commands;

use Drupal\rebuilder\PluginManager\RebuilderManagerInterface;
use Drush\Commands\DrushCommands;
use Drush\Exceptions\CommandFailedException;

/**
 * rebuilder:run Drush command.
 *
 * @see self::runRebuilder()
 */
class RunCommand extends DrushCommands {

  /**
   * Constructor; saves dependencies.
   *
   * @param \Drupal\rebuilder\PluginManager\RebuilderManagerInterface $rebuilderManager
   *   The Rebuilder plug-in manager.
   */
  public function __construct(
    protected readonly RebuilderManagerInterface $rebuilderManager,
  ) {}

  /**
   * Rebuild something cached by Drupal.
   *
   * @command rebuilder:run
   *
   * @aliases rebuilder
   *
   * @param string $rebuilderId
   *   The machine name of the Rebuilder plug-in to run.
   *
   * @param array $options
   *   Drush command options.
   *
   * @option option
   *   Options to pass to the Rebuilder plug-in. See each plug-in for what
   *   options they support, if any.
   *
   * @throws \Drush\Exceptions\CommandFailedException
   *   If the Rebuilder plug-in manager threw an error, with the text of the
   *   error.
   */
  public function runRebuilder(
    string $rebuilderId, array $options = ['option' => []],
  ): void {

    // phpcs:disable Drupal.ControlStructures.ControlSignature.NewlineAfterCloseBrace
    try {

      /** @var \Drupal\Core\StringTranslation\TranslatableMarkup The output from the Rebuilder plug-in. */
      $output = $this->rebuilderManager->runRebuilder(
        $rebuilderId, $options['option'],
      );

      $this->logger()->success((string) $output);

    } catch (\Exception $exception) {

      throw new CommandFailedException($exception->getMessage());

    }
    // phpcs:enable Drupal.ControlStructures.ControlSignature.NewlineAfterCloseBrace

  }

}
