<?php

declare(strict_types=1);

namespace Drupal\rebuilder\Plugin\Rebuilder;

use Drupal\Core\Asset\AssetCollectionOptimizerInterface;
use Drupal\Core\Asset\AssetQueryStringInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\rebuilder\Attribute\Rebuilder;
// phpcs:disable Drupal.Classes.UnusedUseStatement.UnusedUse
use Drupal\rebuilder\Plugin\Rebuilder\RebuilderBase;
use Drupal\rebuilder\PluginManager\RebuilderManagerInterface;
// phpcs:enable Drupal.Classes.UnusedUseStatement.UnusedUse
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Asset rebuilder plug-in.
 */
#[Rebuilder(
  id: 'asset',
  title: new TranslatableMarkup('Asset'),
  description: new TranslatableMarkup('Rebuilds aggregated CSS and JS assets.'),
  aliases: [
    'assets',
  ],
)]
class Asset extends RebuilderBase {

  /**
   * {@inheritdoc}
   *
   * @param \Drupal\Core\Asset\AssetCollectionOptimizerInterface $cssCollectionOptimizer
   *   The Drupal CSS collection optimizer service.
   *
   * @param \Drupal\Core\Asset\AssetCollectionOptimizerInterface $jsCollectionOptimizer
   *   The Drupal JavaScript collection optimizer service.
   *
   * @param \Drupal\Core\Asset\AssetQueryStringInterface $assetQueryString
   *   The Drupal asset query string service.
   *
   * @param \Drupal\rebuilder\PluginManager\RebuilderManagerInterface $rebuilderManager
   *   The Rebuilder plug-in manager.
   */
  public function __construct(
    array $configuration, string $pluginId, array $pluginDefinition,
    TranslationInterface $stringTranslation,
    protected readonly AssetCollectionOptimizerInterface $cssCollectionOptimizer,
    protected readonly AssetCollectionOptimizerInterface $jsCollectionOptimizer,
    protected readonly AssetQueryStringInterface $assetQueryString,
    protected readonly RebuilderManagerInterface $rebuilderManager,
  ) {

    parent::__construct(
      $configuration, $pluginId, $pluginDefinition,
      $stringTranslation,
    );

  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration, $pluginId, $pluginDefinition,
  ) {

    return new static(
      $configuration, $pluginId, $pluginDefinition,
      $container->get('string_translation'),
      $container->get('asset.css.collection_optimizer'),
      $container->get('asset.js.collection_optimizer'),
      $container->get('asset.query_string'),
      $container->get('plugin.manager.rebuilder'),
    );

  }

  /**
   * {@inheritdoc}
   *
   * @see \drupal_flush_all_caches()
   */
  public function rebuild(array $options = []): void {

    $this->cssCollectionOptimizer->deleteAll();
    $this->jsCollectionOptimizer->deleteAll();

    $this->assetQueryString->reset();

    // Library definitions also need to be rebuilt to invalidate relevant cache
    // tags so the new asset URLs actually get attached to rendered output.
    //
    // Note that we're invoking the library rebuilder rather than doing that
    // ourselves here to allow that rebuilder to perform additional tasks.
    $this->rebuilderManager->runRebuilder('library');

    $this->setOutput($this->t('CSS and JS assets rebuilt.'));

  }

}
