<?php

declare(strict_types = 1);

/**
 * Copyright (C) 2025 PRONOVIX GROUP.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 */

namespace Drupal\recipe_code_installer\EventSubscriber;

use Drupal\Core\Recipe\RecipeAppliedEvent;
use Drupal\Core\Utility\Error;
use Drupal\recipe_code_installer\Application\UseCase\InstallBundledModule\Exception\BundledModuleInstallationFailureException;
use Drupal\recipe_code_installer\Application\UseCase\InstallBundledModule\InstallBundledModuleFromRecipeInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Acts on Recipe related events.
 *
 * @internal This class is not part of the module's public programming API.
 */
final class RecipeEventsSubscriber implements EventSubscriberInterface {

  /**
   * Creates a new instance.
   */
  public function __construct(
    private readonly InstallBundledModuleFromRecipeInterface $installer,
    #[Autowire(service: 'logger.channel.recipe_code_installer')]
    private readonly LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [RecipeAppliedEvent::class => 'onRecipeAppliedEvent'];
  }

  /**
   * Copies and installs recipe bundled modules.
   *
   * @param \Drupal\Core\Recipe\RecipeAppliedEvent $event
   *   Event object.
   */
  public function onRecipeAppliedEvent(RecipeAppliedEvent $event): void {
    try {
      ($this->installer)($event->recipe);
    }
    catch (BundledModuleInstallationFailureException $exception) {
      Error::logException($this->logger, $exception, 'Failed to install bundled modules of {recipe} recipe. @message', [
        'recipe' => $event->recipe->name,
      ]);
      // Rethrow to break the install process.
      throw $exception;
    }
  }

}
