<?php

declare(strict_types = 1);

/**
 * Copyright (C) 2025 PRONOVIX GROUP.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 */

namespace Drupal\recipe_code_installer\Application\UseCase\InstallBundledModule;

use Drupal\Core\Extension\ExtensionNameLengthException;
use Drupal\Core\Extension\ExtensionNameReservedException;
use Drupal\Core\Extension\MissingDependencyException;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Extension\ModuleInstallerInterface;
use Drupal\Core\Recipe\Recipe;
use Drupal\Core\Utility\Error;
use Drupal\recipe_code_installer\Application\Port\CustomModulesPathResolverInterface;
use Drupal\recipe_code_installer\Application\Port\Exception\UnableToResolveCustomModulesInstallationPathException;
use Drupal\recipe_code_installer\Application\UseCase\InstallBundledModule\Exception\BundledModuleInstallationFailureException;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Filesystem\Exception\IOException;
use Symfony\Component\Filesystem\Filesystem;

/**
 * Installs a bundled module from a Recipe as a custom module in the project.
 *
 * @internal This class is not part of the module's public programming API.
 */
final class InstallBundledModuleFromRecipe implements InstallBundledModuleFromRecipeInterface {

  /**
   * Creates a new instance.
   */
  public function __construct(
    private readonly ModuleInstallerInterface $moduleInstaller,
    private readonly ModuleHandlerInterface $moduleHandler,
    private readonly ModuleExtensionList $moduleExtensionList,
    private readonly CustomModulesPathResolverInterface $customModulesPathResolver,
    #[Autowire(service: 'logger.channel.recipe_code_installer')]
    private readonly LoggerInterface $logger,
    private readonly Filesystem $filesystem = new Filesystem(),
    private readonly bool $devMode = FALSE,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function __invoke(Recipe $recipe): void {
    $source_dir = $this->findCodePathInRecipe($recipe);
    if ($source_dir === NULL) {
      // No 'code' directory found in the recipe, so nothing to install.
      return;
    }
    $recipe_machine_name = $this->getRecipeMachineName($recipe);

    try {
      $custom_module_path = $this->customModulesPathResolver->getCustomModulesInstallPath();
    }
    catch (UnableToResolveCustomModulesInstallationPathException $e) {
      throw new BundledModuleInstallationFailureException(sprintf('Failed to install the bundled module from the "%s" recipe: unable to resolve the custom modules installation path. %s', $recipe->name, $e->getMessage()), previous: $e);
    }

    $destination_dir = $custom_module_path . DIRECTORY_SEPARATOR . $recipe_machine_name;

    if ($this->filesystem->exists($destination_dir)) {
      $this->logger->info('The bundled module inside the {recipe} recipe has not been {operation} to {path} because it already exists.', [
        'operation' => $this->devMode ? 'symlinked' : 'copied',
        'recipe' => $recipe_machine_name,
        'path' => $destination_dir,
      ]);
    }
    else {
      if ($this->devMode) {
        try {
          $this->filesystem->symlink($this->filesystem->makePathRelative($source_dir, dirname($destination_dir)), $destination_dir);
          $this->logger->debug('The {module} module was successfully symlinked to {path}.', [
            'module' => $recipe_machine_name,
            'path' => $destination_dir,
          ]);
        }
        catch (IOException $e) {
          throw new BundledModuleInstallationFailureException(sprintf('The bundled module inside %s recipe could not be symlinked at destination. %s', $recipe->name, $e->getMessage()), previous: $e);
        }
      }
      else {
        try {
          $this->filesystem->mirror($source_dir, $destination_dir);
          $this->logger->debug('The {module} module was successfully copied to {path}.', [
            'module' => $recipe_machine_name,
            'path' => $destination_dir,
          ]);
        }
        catch (IOException $e) {
          throw new BundledModuleInstallationFailureException(sprintf('The bundled module inside %s recipe could not be copied to destination. %s', $recipe->name, $e->getMessage()), previous: $e);
        }
      }

      $this->ensureNewModuleIsPickedUpByInstaller();
    }

    if ($this->moduleHandler->moduleExists($recipe_machine_name)) {
      $this->logger->info('The bundled module inside the {recipe} recipe is already enabled.', [
        'recipe' => $recipe_machine_name,
        'path' => $destination_dir,
      ]);
    }
    else {
      try {
        $this->moduleInstaller->install([$recipe_machine_name]);
      }
      catch (ExtensionNameLengthException | MissingDependencyException | ExtensionNameReservedException $e) {
        // Attempt to remove the copied files if installation fails.
        try {
          $this->filesystem->remove($destination_dir);
        }
        catch (IOException $ioe) {
          Error::logException(
            $this->logger,
            $ioe,
            'Failed to clean up directory at {path} after the installation of the bundle module inside {recipe} recipe failed. @message',
            [
              'path' => $destination_dir,
              'recipe' => $recipe_machine_name,
            ],
          );
        }

        throw new BundledModuleInstallationFailureException(sprintf('The bundled module inside %s recipe could not be installed. %s', $recipe->name, $e->getMessage()), previous: $e);
      }

      $this->logger->info('The bundled module inside the {recipe} recipe was successfully installed.', [
        'recipe' => $recipe_machine_name,
      ]);
    }
  }

  /**
   * Clears the ED's cache to ensure newly copied modules are detected.
   *
   * ExtensionDiscovery maintains a static cache in its $files property to avoid
   * redundant filesystem scans. When modules are programmatically copied or
   * created, this cache needs to be cleared to force a rescan that will detect
   * the new modules. This cache by design is not flushed when
   * \Drupal\Core\Extension\ExtensionList::reset() is called.
   *
   * The performance impact is minimized by:
   * - Prior to Drupal 11.2.0: ExtensionDiscovery's own file cache
   * - Drupal 11.2.0 and later: The InfoParser's file cache
   *
   * @see https://www.drupal.org/node/3490431
   * @see \Drupal\Core\Extension\ExtensionDiscovery::$files
   * @see \Drupal\Core\Extension\ExtensionDiscovery::scan()
   * @see \Drupal\Core\Extension\ModuleExtensionList::getExtensionDiscovery()
   */
  private function ensureNewModuleIsPickedUpByInstaller(): void {
    $ro = new \ReflectionObject($this->moduleExtensionList);
    $ro->getMethod('getExtensionDiscovery')->setAccessible(TRUE);
    $extensionDiscovery = $ro->getMethod('getExtensionDiscovery')->invoke($this->moduleExtensionList);
    $ro = new \ReflectionObject($extensionDiscovery);
    $ro->getProperty('files')->setAccessible(TRUE);
    $ro->getProperty('files')->setValue(NULL, []);
  }

  /**
   * Retrieves the trimmed path of the given recipe.
   *
   * @param \Drupal\Core\Recipe\Recipe $recipe
   *   The recipe object.
   *
   * @return string
   *   The recipe's path, with any trailing slashes removed.
   */
  private function getRecipePath(Recipe $recipe): string {
    return rtrim($recipe->path, '/');
  }

  /**
   * Finds the path to the bundled code directory within a recipe, if it exists.
   *
   * @param \Drupal\Core\Recipe\Recipe $recipe
   *   The recipe object to inspect.
   *
   * @return string|null
   *   The absolute path of the code inside the recipe or null when the
   *   recipe has no code bundled.
   */
  private function findCodePathInRecipe(Recipe $recipe): ?string {
    if ($this->filesystem->exists($this->getRecipePath($recipe) . '/code')) {
      $abs_path = realpath($this->getRecipePath($recipe) . '/code');
      assert($abs_path !== FALSE);
      return $abs_path;
    }

    return NULL;
  }

  /**
   * Retrieves the machine name of the given recipe.
   *
   * @param \Drupal\Core\Recipe\Recipe $recipe
   *   The recipe object.
   *
   * @return string
   *   The machine name of the recipe.
   */
  private function getRecipeMachineName(Recipe $recipe): string {
    return basename($this->getRecipePath($recipe));
  }

}
