<?php

declare(strict_types = 1);

/**
 * Copyright (C) 2025 PRONOVIX GROUP.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 */

namespace Drupal\recipe_code_installer\Infrastructure\CustomModulesPathResolver;

use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\Core\KeyValueStore\KeyValueStoreInterface;
use Drupal\recipe_code_installer\Application\Port\CustomModulesPathResolverInterface;
use Drupal\recipe_code_installer\Application\Port\Exception\UnableToResolveCustomModulesInstallationPathException;
use Drupal\recipe_code_installer\Application\Port\Exception\UnableToResolveProjectRootException;
use Drupal\recipe_code_installer\Application\Port\ProjectRootPathResolverInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Implementation for resolving custom module installation path.
 *
 * This implementation uses Drupal Scaffold settings in project's composer.json
 * as the source of truth to resolve custom modules installation path.
 *
 * @internal This class is not part of the module's public programming API.
 */
final class ResolveCustomModulesPathsFromProjectComposerJson implements CustomModulesPathResolverInterface {

  /**
   * The key value store.
   */
  private readonly KeyValueStoreInterface $keyValueStore;

  /**
   * Creates a new instance.
   */
  public function __construct(
    #[Autowire(service: 'keyvalue')]
    KeyValueFactoryInterface $keyValueFactory,
    private readonly ProjectRootPathResolverInterface $projectRootPathResolver,
  ) {
    $this->keyValueStore = $keyValueFactory->get('recipe_code_installer.custom_modules_install_path_resolver_from_composer_json');
  }

  /**
   * {@inheritdoc}
   */
  public function getCustomModulesInstallPath(): string {
    try {
      $project_root = $this->projectRootPathResolver->getProjectRootPath();
    }
    catch (UnableToResolveProjectRootException $e) {
      throw new UnableToResolveCustomModulesInstallationPathException('Project root path could not be resolved.', previous: $e);
    }
    return $this->resolveCustomModulePath($project_root);
  }

  /**
   * Resolves custom modules installation path from the project's composer.json.
   *
   * @param string $project_root
   *   The project root.
   *
   * @return string
   *   Custom module installation path.
   *
   * @throws \Drupal\recipe_code_installer\Application\Port\Exception\UnableToResolveCustomModulesInstallationPathException
   */
  private function resolveCustomModulePath(string $project_root): string {
    $composer_lock_path = $project_root . DIRECTORY_SEPARATOR . 'composer.lock';
    if (!@is_readable($composer_lock_path)) {
      throw new UnableToResolveCustomModulesInstallationPathException(sprintf("The project's composer.lock not found at %s path.", $composer_lock_path));
    }

    $fp = @fopen($composer_lock_path, 'rb');
    if (!$fp) {
      // Should not happen after is_readable() returned true.
      throw new UnableToResolveCustomModulesInstallationPathException(sprintf("Failed to open project's composer.lock for read at %s path.", $composer_lock_path));
    }

    // Read the first 2KB (safe enough to catch the hash in almost all cases).
    $partial = fread($fp, 2048);
    if ($partial === FALSE) {
      throw new UnableToResolveCustomModulesInstallationPathException(sprintf("Failed to read 2KB from project's composer.lock at %s path.", $composer_lock_path));
    }
    if (fclose($fp) === FALSE) {
      throw new UnableToResolveCustomModulesInstallationPathException(sprintf("Failed to close project's composer.lock after read at %s path.", $composer_lock_path));
    }

    if (preg_match('/"content-hash"\s*:\s*"([^"]+)"/', $partial, $matches) === 1) {
      $hash = substr($matches[1], 0, 32);
      $cid = "recipe_code_installer:composer_data:$hash";
    }
    else {
      throw new UnableToResolveCustomModulesInstallationPathException(sprintf("Failed to extract content-hash from project's composer.lock after read at %s path.", $composer_lock_path));
    }

    $cached_path = $this->keyValueStore->get($cid);
    if ($cached_path) {
      return $cached_path;
    }

    $composer_json_path = $project_root . DIRECTORY_SEPARATOR . 'composer.json';
    if (!@is_readable($composer_json_path)) {
      throw new UnableToResolveCustomModulesInstallationPathException(sprintf("The project's composer.json not found at %s path.", $composer_json_path));
    }

    error_clear_last();
    $content = @file_get_contents($composer_json_path);
    if ($content === FALSE) {
      // Should not happen after is_readable() returned true.
      throw new UnableToResolveCustomModulesInstallationPathException(sprintf("Failed to read the project's composer.json at %s path. %s", $composer_json_path, error_get_last()['message'] ?? ''));
    }

    try {
      $data = json_decode($content, TRUE, flags: JSON_THROW_ON_ERROR);
    }
    catch (\JsonException $e) {
      throw new UnableToResolveCustomModulesInstallationPathException(sprintf("Failed to decode project's composer.json at %s path. %s", $composer_json_path, $e->getMessage()));
    }

    $path = $project_root . DIRECTORY_SEPARATOR . $this->getCustomModulePathFromComposerData($data);
    $this->keyValueStore->set($cid, $path);

    return $path;
  }

  /**
   * Extracts the path for custom module installation from composer data.
   *
   * @param array $data
   *   Content of composer.json as array.
   *
   * @return string
   *   The installation path.
   *
   * @phpstan-ignore missingType.iterableValue
   */
  protected function getCustomModulePathFromComposerData(array $data): string {
    $default_path = 'web/modules/custom';

    if (!isset($data['extra']['installer-paths'])) {
      return $default_path;
    }

    foreach ($data['extra']['installer-paths'] as $path => $types) {
      if (in_array('type:drupal-custom-module', $types, TRUE)) {
        // @phpstan-ignore return.type
        return preg_replace('#/\{\$name}(/)?$#', '', trim($path, '/'));
      }
    }

    return $default_path;
  }

}
