# Recipes Helper

A generic Drupal module that provides Drush commands to export and import entity bundles (content types, paragraph types, media types, vocabularies, custom block types) using [Drupal recipes](https://www.drupal.org/docs/extending-drupal/drupal-recipes) for easy migration between sites.

## Features

- **Generic entity bundle export** - Works with any bundleable entity type
- **Supported entity types:**
  - Node types (content types)
  - Paragraph types
  - Media types
  - Taxonomy vocabularies
  - Block content types
- **Advanced dependency detection:**
  - Detects and exports referenced media types and taxonomies
  - Discovers edge case configs (field_validation, conditional_fields, etc.)
  - Bundle-level detection prevents false positives from Views, Search API, etc.
  - Automatic module dependency inference from config prefixes
- **Complete configuration export** - Exports fields, displays, form modes, and all related configurations
- **Recipe generation** - Automatically creates recipe.yml files
- **Documentation** - Generates README for each export
- **Safe import by default** - Import recipes with automatic module installation
- **Force overwrite option** - Optional `--force` flag to overwrite existing configs
- **Batch operations** - Import all recipes from a directory in one command

## Installation

1. Enable the module:
   ```bash
   ddev drush en recipes_helper -y
   ```

2. Clear cache:
   ```bash
   ddev drush cr
   ```

## Commands Overview

| Command | Alias | Description |
|---------|-------|-------------|
| `recipe:list:types` | `rlt` | List all supported entity types |
| `recipe:list` | `rlb` | List all bundles for an entity type |
| `recipe:export` | `rex` | Export a single bundle to a recipe |
| `recipe:export:all` | `rexa` | Export all bundles of an entity type |
| `recipe:import` | `rim` | Import a recipe with module installation |
| `recipe:import:all` | `rima` | Import all recipes from a directory |

## Commands

### 1. List Supported Entity Types

```bash
ddev drush recipe:list:types
# or short alias
ddev drush rlt
```

**Output:**
```
Supported Entity Types:
----------------------------------------------------------
  node
  paragraph
  media
  taxonomy_term
  block_content
----------------------------------------------------------
```

### 2. List Bundles for an Entity Type

```bash
ddev drush recipe:list <entity_type>
# or short alias
ddev drush rls <entity_type>
```

**Examples:**
```bash
# List all content types
ddev drush rls node

# List all paragraph types
ddev drush rls paragraph

# List all media types
ddev drush rls media

# List all vocabularies
ddev drush rls taxonomy_term

# List all custom block types
ddev drush rls block_content
```

### 3. Export a Bundle

```bash
ddev drush recipe:export <entity_type> <bundle>
# or short alias
ddev drush rex <entity_type> <bundle>
```

**Examples:**
```bash
# Export a content type
ddev drush rex node article

# Export a paragraph type
ddev drush rex paragraph hero_banner

# Export a media type
ddev drush rex media image

# Export a taxonomy vocabulary
ddev drush rex taxonomy_term tags

# Export a custom block type
ddev drush rex block_content basic

# Export to custom destination
ddev drush rex node article --destination=/tmp/my_recipes
```

### 4. Export All Bundles of an Entity Type

```bash
ddev drush recipe:export:all <entity_type>
# or short alias
ddev drush rexa <entity_type>
```

This command exports **all bundles** of a specific entity type to separate recipe folders.

**Examples:**
```bash
# Export all paragraph types
ddev drush rexa paragraph

# Export all media types
ddev drush rexa media

# Export all content types
ddev drush rexa node

# Export to custom destination
ddev drush rexa paragraph --destination=recipes/paragraphs
```

**Output Structure:**
```
recipes/custom/
├── hero_banner/
│   ├── recipe.yml
│   ├── config/
│   └── README.md
├── text_block/
│   ├── recipe.yml
│   ├── config/
│   └── README.md
└── banner/
    ├── recipe.yml
    ├── config/
    └── README.md
```

### 5. Import a Recipe

```bash
ddev drush recipe:import <recipe_path> [--force]
# or short alias
ddev drush rim <recipe_path> [--force]
```

This command will:
1. **Install all required modules** listed in recipe.yml
2. **Import configurations** (skips existing configs by default, use `--force` to overwrite)
3. **Rebuild cache**

**Examples:**
```bash
# Import a recipe (skips existing configs)
ddev drush rim sites/my-site/recipes/types/node/page

# Force overwrite existing configs
ddev drush rim recipes/custom/accordion --force

# Import to a different site using alias
ddev drush @my-site rim recipes/custom/hero_banner
```

**Options:**
- `--force` - Overwrite existing configurations. Without this flag, the command will skip any configs that already exist.

**Important Notes:**
- By default, existing configurations are **NOT overwritten** (safe mode)
- Use `--force` to overwrite existing configurations
- All required modules will be automatically installed
- The import bypasses standard Drupal recipe validation
- Use `--force` with caution on production sites

### 6. Import All Recipes from a Directory

```bash
ddev drush recipe:import:all <recipes_directory> [--force]
# or short alias
ddev drush rima <recipes_directory> [--force]
```

This command will:
1. **Scan the directory** for all recipe folders (containing recipe.yml)
2. **Import each recipe** in sequence
3. **Install required modules** and import configurations for each recipe
4. **Provide summary** of successful and failed imports

**Examples:**
```bash
# Import all paragraph recipes
ddev drush rima sites/my-site/recipes/types/paragraph

# Import all node type recipes with force overwrite
ddev drush rima sites/my-site/recipes/types/node --force

# Import to a different site using alias
ddev drush @my-site rima recipes/custom
```

**Options:**
- `--force` - Overwrite existing configurations in all recipes.

**Important Notes:**
- Processes all subdirectories that contain a `recipe.yml` file
- Each recipe is imported independently
- Failed recipes don't stop the process - continues with remaining recipes
- Final summary shows total imported vs failed

## What Gets Exported

For any entity bundle, the export includes **only what's needed**:

- ✅ Bundle/type definition (node type, paragraph type, etc.)
- ✅ **Only the field storage configurations used by this bundle** (not all entity type storages)
- ✅ All field instance configurations for the bundle
- ✅ All form display configurations (default, custom modes)
- ✅ All view display configurations (default, teaser, custom modes)
- ✅ Language and translation settings
- ✅ **Automatically detected dependencies:**
  - Referenced media types (with their fields and displays)
  - Referenced taxonomy vocabularies (with their fields)
  - Required field types (image, file, etc.)
  - All necessary modules

### Smart Dependency Detection

The module uses multiple strategies to ensure complete and accurate exports:

#### 1. Smart Field Storage Export
The module **only exports field storage that is actually used** by the bundle. For example:
- A paragraph type with 2 fields exports only 2 field storage configs
- Not all 68+ field storage configs for all paragraph types

#### 2. Bundle-Referencing Config Discovery
Automatically finds configs that reference the bundle but aren't stored within standard bundle configs:
- **Method 1:** Checks for `entity_type` and `bundle` keys
- **Method 2:** Checks for `targetEntityType` and `bundle` keys
- **Method 3:** Checks naming patterns like `<module>.<type>.<entity_type>_<bundle>`

Uses **bundle-level detection only** to avoid false positives from Views, Search API indexes, and other configs that may reference field names.

**Real-world examples:**
- `field_validation.rule_set.paragraph_hero_banner` - Field validation rules
- `conditional_fields.config.paragraph.accordion` - Conditional field configurations
- Any contrib module configs that reference your bundle

#### 3. Automatic Module Detection
Infers required modules from config name prefixes:
- `field_validation.*` → requires `field_validation` module
- `conditional_fields.*` → requires `conditional_fields` module
- Works with any contrib module following Drupal naming conventions

This comprehensive approach ensures that **all related configurations are captured**, preventing import failures due to missing configs or modules.

## Output Structure

Each export creates a directory structure like this:

```
recipes/custom/<bundle>/
├── recipe.yml              # Recipe definition with dependencies
├── config/                 # All configuration files
│   ├── node.type.article.yml
│   ├── field.field.node.article.*.yml
│   ├── core.entity_form_display.node.article.*.yml
│   ├── core.entity_view_display.node.article.*.yml
│   └── ... (all related configs)
└── README.md               # Documentation for the export
```

## Example Workflow

```bash
# 1. List available entity types
ddev drush rlt

# 2. List bundles for a specific entity type
ddev drush rls node

# Output:
# Available bundles for 'node':
# ----------------------------------------------------------
#   article - Article
#   page - Basic page
#   landing_page - Landing Page
# ----------------------------------------------------------

# 3. Export a specific bundle
ddev drush rex node article

# Output:
# Exporting node bundle: article
# Destination: /var/www/html/recipes/custom/article
#   Exported: node.type.article
#   Exported: field.storage.node.body
#   Exported: field.field.node.article.body
#   Exported: field.field.node.article.field_image
#   ...
# Exporting referenced media type: image
#   Exported: media.type.image
#   ...
# Created recipe.yml
# Created README.md
# Export completed successfully!
# Exported 42 configuration files.
```

## Importing Recipes on Target Site

### Method 1: Using Drupal Recipe Command (Recommended)

```bash
# Copy the exported directory to target site, then:
php docroot/core/scripts/drupal recipe recipes/custom/article
```

### Method 2: Using Drush (if available)

```bash
ddev drush recipe recipes/custom/article
```

### Method 3: Manual Config Import

```bash
# Copy config files to sync directory
cp recipes/custom/article/config/*.yml config/sync/

# Import configurations
ddev drush config:import --partial
```

## Supported Entity Types

| Entity Type | Description | Example Bundles |
|-------------|-------------|-----------------|
| `node` | Content types | article, page, landing_page |
| `paragraph` | Paragraph types | hero_banner, text_block, image_gallery |
| `media` | Media types | image, video, document, audio |
| `taxonomy_term` | Vocabularies | tags, categories, topics |
| `block_content` | Custom blocks | basic, banner, promo |

## Use Cases

### Migrating Content Types Between Sites

```bash
# Export from source site
ddev drush rex node landing_page

# Copy to target site and import
php docroot/core/scripts/drupal recipe recipes/custom/node_landing_page_export
```

### Creating Reusable Components

```bash
# Export a paragraph type with all dependencies
ddev drush rex paragraph call_to_action

# Share with other projects
# The recipe includes all media types and taxonomies it references
```

### Building Site Distributions

```bash
# Export all your custom entity bundles
ddev drush rex node article
ddev drush rex node landing_page
ddev drush rex paragraph hero_banner
ddev drush rex media video

# Package all recipes for distribution
```

### Environment Synchronization

```bash
# Export from dev environment
ddev drush rex node product --destination=../recipes/prod

# Deploy to production
git add recipes/prod/node_product_export
git commit -m "Add product content type recipe"

# On production
git pull
php docroot/core/scripts/drupal recipe recipes/prod/node_product_export
```

## Advanced Usage

### Export Multiple Bundles

```bash
# Create a script to export multiple bundles
for bundle in article page landing_page; do
  ddev drush rex node $bundle
done
```

### Custom Destination

```bash
# Export to a specific location
ddev drush rex paragraph hero_banner --destination=/tmp/hero_export

# Export to a git-tracked directory
ddev drush rex node article --destination=../recipes/production
```

### Verify Before Import

```bash
# Review the recipe before importing
cat recipes/custom/node_article_export/recipe.yml

# Check what will be imported
ls -la recipes/custom/node_article_export/config/

# Read the documentation
cat recipes/custom/node_article_export/README.md
```

## Advanced Dependency Detection

The module uses a comprehensive multi-strategy approach to detect and export all related configurations:

### 1. Entity Reference Detection
**Media References:**
- Detects media type (image, video, document, etc.)
- Exports complete media type configuration with fields and displays
- Adds media module to dependencies

**Taxonomy References:**
- Detects vocabulary references
- Exports vocabulary configuration with fields
- Adds taxonomy module to dependencies

### 2. Field Type Detection
Automatically adds required modules based on field types:
- Image fields → `image` module
- File fields → `file` module
- Entity reference fields → appropriate modules

### 3. Edge Case Config Discovery
Scans **all system configurations** to find configs that reference your bundle:

**Bundle-Level Detection (3 Methods):**
- **Method 1:** Configs with `entity_type` and `bundle` keys
- **Method 2:** Configs with `targetEntityType` and `bundle` keys
- **Method 3:** Configs matching naming pattern `<module>.<type>.<entity_type>_<bundle>`

Uses **bundle-level detection only** to avoid false positives. Field names are not unique (multiple bundles can use `field_text`), so field-level detection would incorrectly include unrelated Views, Search API indexes, and other configs.

**Caught by this strategy:**
- `field_validation.rule_set.*` - Field validation rules
- `conditional_fields.config.*` - Conditional field dependencies
- `layout_paragraphs.layout.*` - Layout paragraphs configurations
- Any contrib module configs that reference your specific bundle

### 4. Module Dependency Inference
Automatically infers required modules from config name prefixes:
- Scans all exported configs
- Extracts module name from config prefix (e.g., `field_validation` from `field_validation.rule_set.*`)
- Verifies module exists and adds to dependencies
- Prevents import failures due to missing modules

This multi-layered approach ensures **complete bundle configuration coverage**, catching even complex edge cases where contrib modules create configs outside standard Drupal patterns, while avoiding false positives from unrelated configurations.

## Troubleshooting

### "Unsupported entity type"
**Solution**: Run `ddev drush rlt` to see supported entity types. Only bundleable entity types are supported.

### "Bundle does not exist"
**Solution**: Run `ddev drush rls <entity_type>` to see available bundles. Check the machine name, not the label.

### Recipe import fails
**Solution**:
1. Ensure all required modules are installed on target site
2. Check module versions are compatible
3. Try partial config import: `ddev drush config:import --partial --source=path/to/config`

### Dependencies not detected
**Solution**: The module automatically detects media and taxonomy references. If something is missing:
1. Check the generated recipe.yml
2. Manually add missing modules to the `install` section
3. Re-export if needed

## Extending the Module

### Adding New Entity Types

To add support for additional entity types, edit the `$entityConfigMap` array in `RecipeExportCommands.php`:

```php
protected $entityConfigMap = [
  'your_entity_type' => [
    'type_prefix' => 'your_entity.type',
    'field_prefix' => 'field.field.your_entity',
    'form_display_prefix' => 'core.entity_form_display.your_entity',
    'view_display_prefix' => 'core.entity_view_display.your_entity',
    'language_prefix' => 'language.content_settings.your_entity',
  ],
];
```

## Requirements

- Drupal 10.x or 11.x
- Drush 12 or higher (Drush 13 recommended)
- Write permissions to recipes directory

For more information about Drupal recipes, see the [official Drupal recipes documentation](https://www.drupal.org/docs/extending-drupal/drupal-recipes).

## Contributing

To add new features or entity type support:

1. Edit `RecipeExportCommands.php`
2. Add new entity type mappings
3. Update this README
4. Test exports and imports

## License

GPL-2.0-or-later

## Credits

Co-authored with Claude (Anthropic's AI assistant).

Created for flexible Drupal configuration management and site-to-site migrations.
