<?php

namespace Drupal\recipes_helper\Commands;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\StorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drush\Commands\DrushCommands;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Yaml\Yaml;

/**
 * Drush commands for exporting and importing entity types and bundles using recipes.
 */
class RecipeCommands extends DrushCommands {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The config storage.
   *
   * @var \Drupal\Core\Config\StorageInterface
   */
  protected $configStorage;

  /**
   * The filesystem.
   *
   * @var \Symfony\Component\Filesystem\Filesystem
   */
  protected $filesystem;

  /**
   * Mapping of entity types to their config prefixes.
   *
   * @var array
   */
  protected $entityConfigMap = [
    'node' => [
      'type_prefix' => 'node.type',
      'field_prefix' => 'field.field.node',
      'form_display_prefix' => 'core.entity_form_display.node',
      'view_display_prefix' => 'core.entity_view_display.node',
      'language_prefix' => 'language.content_settings.node',
    ],
    'paragraph' => [
      'type_prefix' => 'paragraphs.paragraphs_type',
      'field_prefix' => 'field.field.paragraph',
      'form_display_prefix' => 'core.entity_form_display.paragraph',
      'view_display_prefix' => 'core.entity_view_display.paragraph',
      'language_prefix' => 'language.content_settings.paragraph',
    ],
    'media' => [
      'type_prefix' => 'media.type',
      'field_prefix' => 'field.field.media',
      'form_display_prefix' => 'core.entity_form_display.media',
      'view_display_prefix' => 'core.entity_view_display.media',
      'language_prefix' => 'language.content_settings.media',
    ],
    'taxonomy_term' => [
      'type_prefix' => 'taxonomy.vocabulary',
      'field_prefix' => 'field.field.taxonomy_term',
      'form_display_prefix' => 'core.entity_form_display.taxonomy_term',
      'view_display_prefix' => 'core.entity_view_display.taxonomy_term',
      'language_prefix' => 'language.content_settings.taxonomy_term',
    ],
    'block_content' => [
      'type_prefix' => 'block_content.type',
      'field_prefix' => 'field.field.block_content',
      'form_display_prefix' => 'core.entity_form_display.block_content',
      'view_display_prefix' => 'core.entity_view_display.block_content',
      'language_prefix' => 'language.content_settings.block_content',
    ],
  ];

  /**
   * Constructs a RecipeExportCommands object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Config\StorageInterface $config_storage
   *   The config storage.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    ConfigFactoryInterface $config_factory,
    StorageInterface $config_storage
  ) {
    parent::__construct();
    $this->entityTypeManager = $entity_type_manager;
    $this->configFactory = $config_factory;
    $this->configStorage = $config_storage;
    $this->filesystem = new Filesystem();
  }

  /**
   * Export an entity bundle to a recipe.
   *
   * @param string $entity_type
   *   The entity type (node, paragraph, media, taxonomy_term, block_content).
   * @param string $bundle
   *   The machine name of the bundle to export.
   * @param array $options
   *   The command options.
   *
   * @command recipes-helper:export
   * @option destination Destination directory for the recipe.
   * @aliases rhex
   * @usage recipes-helper:export node article
   *   Export the article node type to a recipe.
   * @usage recipes-helper:export paragraph hero_banner
   *   Export the hero_banner paragraph type to a recipe.
   * @usage recipes-helper:export media image --destination=/tmp/recipes
   *   Export to a custom destination.
   */
  public function exportBundle($entity_type, $bundle, array $options = ['destination' => NULL]) {
    // Validate entity type.
    if (!isset($this->entityConfigMap[$entity_type])) {
      $this->logger()->error("Unsupported entity type: $entity_type. Supported types: " . implode(', ', array_keys($this->entityConfigMap)));
      return DrushCommands::EXIT_FAILURE;
    }

    $destination = $options['destination'] ?? 'recipes/custom/' . $bundle;

    // Convert to absolute path if relative.
    if (!$this->filesystem->isAbsolutePath($destination)) {
      $destination = DRUPAL_ROOT . '/../' . $destination;
    }

    $config_dir = $destination . '/config';
    $config_map = $this->entityConfigMap[$entity_type];

    $this->output()->writeln("<info>Exporting $entity_type bundle: $bundle</info>");
    $this->output()->writeln("Destination: $destination");

    // Create directories.
    $this->filesystem->mkdir($config_dir);

    // Verify bundle exists.
    if (!$this->bundleExists($entity_type, $bundle)) {
      $this->logger()->error("Bundle '$bundle' does not exist for entity type '$entity_type'.");
      return DrushCommands::EXIT_FAILURE;
    }

    $exported_configs = [];
    $dependencies = [
      'modules' => $this->getBaseModules($entity_type),
      'media_types' => [],
      'vocabularies' => [],
    ];

    // Export bundle/type definition.
    $this->exportConfig($config_map['type_prefix'] . '.' . $bundle, $config_dir, $exported_configs);

    // First, get all field instances for this bundle to determine which storage is needed.
    $field_configs = $this->configStorage->listAll($config_map['field_prefix'] . '.' . $bundle);
    $used_field_storages = [];

    foreach ($field_configs as $config_name) {
      $this->exportConfig($config_name, $config_dir, $exported_configs);

      // Analyze dependencies.
      $config_data = $this->configStorage->read($config_name);
      $this->analyzeDependencies($config_data, $dependencies);

      // Track which field storage this field uses.
      if (isset($config_data['field_name'])) {
        $storage_name = "field.storage.$entity_type." . $config_data['field_name'];
        $used_field_storages[$storage_name] = TRUE;
      }
    }

    // Export only the field storage configurations that are actually used by this bundle.
    foreach (array_keys($used_field_storages) as $storage_config_name) {
      $this->exportConfig($storage_config_name, $config_dir, $exported_configs);
    }

    // Export form displays.
    $form_displays = $this->configStorage->listAll($config_map['form_display_prefix'] . '.' . $bundle);
    foreach ($form_displays as $config_name) {
      $this->exportConfig($config_name, $config_dir, $exported_configs);
    }

    // Export view displays.
    $view_displays = $this->configStorage->listAll($config_map['view_display_prefix'] . '.' . $bundle);
    foreach ($view_displays as $config_name) {
      $this->exportConfig($config_name, $config_dir, $exported_configs);
    }

    // Export language settings.
    $this->exportConfig($config_map['language_prefix'] . '.' . $bundle, $config_dir, $exported_configs);

    // Export referenced entity types.
    $this->exportReferencedEntities($dependencies, $config_dir, $exported_configs);

    // Export configs that reference this bundle.
    $this->exportBundleReferencingConfigs($entity_type, $bundle, $config_dir, $exported_configs);

    // Scan all exported configs for module dependencies.
    $this->scanConfigDependencies($config_dir, $dependencies);

    // Create recipe.yml.
    $this->createRecipeYml($entity_type, $bundle, $destination, $dependencies);

    // Create README.
    $this->createReadme($entity_type, $bundle, $destination, $exported_configs, $dependencies);

    $this->output()->writeln("\n<info>Export completed successfully!</info>");
    $this->output()->writeln("Exported " . count($exported_configs) . " configuration files.");
    $this->output()->writeln("Recipe created at: $destination");

    return DrushCommands::EXIT_SUCCESS;
  }

  /**
   * List all available bundles for an entity type.
   *
   * @param string $entity_type
   *   The entity type (node, paragraph, media, taxonomy_term, block_content).
   *
   * @command recipes-helper:list
   * @aliases rhlb
   * @usage recipes-helper:list node
   *   List all available node types.
   * @usage recipes-helper:list paragraph
   *   List all available paragraph types.
   */
  public function listBundles($entity_type) {
    // Validate entity type.
    if (!isset($this->entityConfigMap[$entity_type])) {
      $this->logger()->error("Unsupported entity type: $entity_type. Supported types: " . implode(', ', array_keys($this->entityConfigMap)));
      return DrushCommands::EXIT_FAILURE;
    }

    $bundles = $this->getBundles($entity_type);

    if (empty($bundles)) {
      $this->output()->writeln("<comment>No bundles found for entity type: $entity_type</comment>");
      return DrushCommands::EXIT_SUCCESS;
    }

    $this->output()->writeln("\n<info>Available bundles for '$entity_type':</info>");
    $this->output()->writeln(str_repeat('-', 60));

    foreach ($bundles as $id => $label) {
      $this->output()->writeln("  <comment>$id</comment> - $label");
    }

    $this->output()->writeln(str_repeat('-', 60));
    $this->output()->writeln("\nTo export a bundle, use:");
    $this->output()->writeln("  <comment>drush recipes-helper:export $entity_type <bundle_name></comment>");

    return DrushCommands::EXIT_SUCCESS;
  }

  /**
   * List all supported entity types.
   *
   * @command recipes-helper:list:types
   * @aliases rhlt
   * @usage recipes-helper:list:types
   *   List all supported entity types.
   */
  public function listEntityTypes() {
    $this->output()->writeln("\n<info>Supported Entity Types:</info>");
    $this->output()->writeln(str_repeat('-', 60));

    foreach (array_keys($this->entityConfigMap) as $entity_type) {
      $this->output()->writeln("  <comment>$entity_type</comment>");
    }

    $this->output()->writeln(str_repeat('-', 60));
    $this->output()->writeln("\nTo list bundles for an entity type:");
    $this->output()->writeln("  <comment>drush recipes-helper:list <entity_type></comment>");
    $this->output()->writeln("\nTo export a bundle:");
    $this->output()->writeln("  <comment>drush recipes-helper:export <entity_type> <bundle></comment>");

    return DrushCommands::EXIT_SUCCESS;
  }

  /**
   * Export all bundles of an entity type to separate recipes.
   *
   * @param string $entity_type
   *   The entity type (node, paragraph, media, taxonomy_term, block_content).
   * @param array $options
   *   The command options.
   *
   * @command recipes-helper:export:all
   * @option destination Base destination directory for all recipes.
   * @aliases rhexa
   * @usage recipes-helper:export:all paragraph
   *   Export all paragraph types to separate recipes.
   * @usage recipes-helper:export:all node --destination=recipes/content_types
   *   Export all node types to a custom directory.
   */
  public function exportAllBundles($entity_type, array $options = ['destination' => NULL]) {
    // Validate entity type.
    if (!isset($this->entityConfigMap[$entity_type])) {
      $this->logger()->error("Unsupported entity type: $entity_type. Supported types: " . implode(', ', array_keys($this->entityConfigMap)));
      return DrushCommands::EXIT_FAILURE;
    }

    $bundles = $this->getBundles($entity_type);

    if (empty($bundles)) {
      $this->output()->writeln("<comment>No bundles found for entity type: $entity_type</comment>");
      return DrushCommands::EXIT_SUCCESS;
    }

    $base_destination = $options['destination'] ?? 'recipes/custom';
    $total_bundles = count($bundles);
    $exported_count = 0;
    $failed_count = 0;

    $this->output()->writeln("\n<info>Exporting all $entity_type bundles ($total_bundles total)...</info>");
    $this->output()->writeln(str_repeat('=', 70));

    foreach ($bundles as $bundle_id => $bundle_label) {
      $this->output()->writeln("\n<comment>[$bundle_id]</comment> $bundle_label");
      $this->output()->writeln(str_repeat('-', 70));

      // Set destination for this bundle.
      $bundle_destination = $base_destination . '/' . $bundle_id;

      // Export the bundle.
      $result = $this->exportBundle($entity_type, $bundle_id, ['destination' => $bundle_destination]);

      if ($result === DrushCommands::EXIT_SUCCESS) {
        $exported_count++;
        $this->output()->writeln("<info>✓ Successfully exported $bundle_id</info>");
      }
      else {
        $failed_count++;
        $this->output()->writeln("<error>✗ Failed to export $bundle_id</error>");
      }
    }

    // Summary.
    $this->output()->writeln("\n" . str_repeat('=', 70));
    $this->output()->writeln("<info>Export Summary:</info>");
    $this->output()->writeln("  Total bundles: $total_bundles");
    $this->output()->writeln("  <info>Successfully exported: $exported_count</info>");
    if ($failed_count > 0) {
      $this->output()->writeln("  <error>Failed: $failed_count</error>");
    }
    $this->output()->writeln("\nAll recipes exported to: $base_destination/");

    return $failed_count > 0 ? DrushCommands::EXIT_FAILURE : DrushCommands::EXIT_SUCCESS;
  }

  /**
   * Import configurations from a recipe directory.
   *
   * This command will:
   * 1. Install all required modules listed in recipe.yml
   * 2. Import configurations (optionally overwriting existing ones with --force)
   * 3. Rebuild cache
   *
   * @param string $recipe_path
   *   Path to the recipe directory (relative to docroot or absolute).
   * @param array $options
   *   The command options.
   *
   * @command recipes-helper:import
   * @option force Force overwrite existing configurations.
   * @aliases rhim
   * @usage recipes-helper:import sites/my-site/recipes/types/node/page
   *   Import the page node type recipe, installing required modules and configs.
   * @usage recipes-helper:import recipes/custom/accordion --force
   *   Import accordion paragraph recipe, overwriting existing configs.
   */
  public function importRecipe($recipe_path, array $options = ['force' => FALSE]) {
    // Convert to absolute path if relative.
    if (!$this->filesystem->isAbsolutePath($recipe_path)) {
      $recipe_path = DRUPAL_ROOT . '/' . $recipe_path;
    }

    // Validate recipe directory exists.
    if (!is_dir($recipe_path)) {
      $this->logger()->error("Recipe directory not found: $recipe_path");
      return DrushCommands::EXIT_FAILURE;
    }

    $recipe_yml = $recipe_path . '/recipe.yml';
    $config_dir = $recipe_path . '/config';

    if (!file_exists($recipe_yml)) {
      $this->logger()->error("recipe.yml not found in: $recipe_path");
      return DrushCommands::EXIT_FAILURE;
    }

    if (!is_dir($config_dir)) {
      $this->logger()->error("config directory not found in: $recipe_path");
      return DrushCommands::EXIT_FAILURE;
    }

    // Parse recipe.yml.
    $recipe_data = Yaml::parseFile($recipe_yml);
    $this->output()->writeln("\n<info>Importing recipe: " . ($recipe_data['name'] ?? 'Unknown') . "</info>");
    $this->output()->writeln("<comment>Description: " . ($recipe_data['description'] ?? 'N/A') . "</comment>");

    // Install required modules.
    if (isset($recipe_data['install']) && !empty($recipe_data['install'])) {
      $modules_to_install = array_filter($recipe_data['install'], function($module) {
        return !\Drupal::moduleHandler()->moduleExists($module);
      });

      if (!empty($modules_to_install)) {
        $this->output()->writeln("\n<info>Installing " . count($modules_to_install) . " required modules...</info>");
        foreach ($modules_to_install as $module) {
          $this->output()->writeln("  - $module");
        }

        try {
          \Drupal::service('module_installer')->install($modules_to_install);
          $this->output()->writeln("<info>✓ Modules installed successfully</info>");
        }
        catch (\Exception $e) {
          $this->logger()->error("Failed to install modules: " . $e->getMessage());
          return DrushCommands::EXIT_FAILURE;
        }
      }
      else {
        $this->output()->writeln("\n<comment>All required modules are already installed</comment>");
      }
    }

    // Import configurations using ConfigImporter.
    $this->output()->writeln("\n<info>Importing configurations...</info>");

    try {
      $config_storage = \Drupal::service('config.storage');
      $source_storage = new \Drupal\Core\Config\FileStorage($config_dir);

      // Get all config files from source.
      $config_names = $source_storage->listAll();
      $this->output()->writeln("Found " . count($config_names) . " configuration files");

      $imported_count = 0;
      $failed_count = 0;
      $skipped_count = 0;

      foreach ($config_names as $config_name) {
        try {
          $config_data = $source_storage->read($config_name);
          if ($config_data === FALSE) {
            $this->logger()->warning("Could not read: $config_name");
            $failed_count++;
            continue;
          }

          // Check if config exists and --force is not set.
          $config = $this->configFactory->getEditable($config_name);
          $config_exists = !$config->isNew();

          if ($config_exists && !$options['force']) {
            $this->output()->writeln("  ⊗ Skipped (exists): $config_name");
            $skipped_count++;
            continue;
          }

          $config->setData($config_data)->save(TRUE);
          $status = $config_exists ? 'Overwritten' : 'Imported';
          $this->output()->writeln("  ✓ $status: $config_name");
          $imported_count++;
        }
        catch (\Exception $e) {
          $this->logger()->error("Failed to import $config_name: " . $e->getMessage());
          $failed_count++;
        }
      }

      // Clear all caches after import.
      $this->output()->writeln("\n<info>Rebuilding cache...</info>");
      drupal_flush_all_caches();

      $this->output()->writeln("\n" . str_repeat('=', 70));
      $this->output()->writeln("<info>Import Summary:</info>");
      $this->output()->writeln("  Successfully imported: $imported_count");
      if ($skipped_count > 0) {
        $this->output()->writeln("  <comment>Skipped (already exist): $skipped_count</comment>");
      }
      if ($failed_count > 0) {
        $this->output()->writeln("  <error>Failed: $failed_count</error>");
      }
      $this->output()->writeln("\n<info>Recipe import completed!</info>");

      if ($skipped_count > 0 && !$options['force']) {
        $this->output()->writeln("<comment>Tip: Use --force to overwrite existing configurations</comment>");
      }

      return $failed_count > 0 ? DrushCommands::EXIT_FAILURE : DrushCommands::EXIT_SUCCESS;
    }
    catch (\Exception $e) {
      $this->logger()->error("Import failed: " . $e->getMessage());
      return DrushCommands::EXIT_FAILURE;
    }
  }

  /**
   * Import all recipes from a directory.
   *
   * This command will:
   * 1. Scan the directory for all recipe folders (containing recipe.yml)
   * 2. Import each recipe in sequence
   * 3. Install required modules and import configurations for each
   *
   * @param string $recipes_dir
   *   Path to the directory containing recipe folders.
   * @param array $options
   *   The command options.
   *
   * @command recipes-helper:import:all
   * @option force Force overwrite existing configurations.
   * @aliases rhima
   * @usage recipes-helper:import:all sites/my-site/recipes/types/paragraph
   *   Import all paragraph recipes from a directory.
   * @usage recipes-helper:import:all recipes/custom --force
   *   Import all recipes with force overwrite.
   */
  public function importAllRecipes($recipes_dir, array $options = ['force' => FALSE]) {
    // Convert to absolute path if relative.
    if (!$this->filesystem->isAbsolutePath($recipes_dir)) {
      $recipes_dir = DRUPAL_ROOT . '/' . $recipes_dir;
    }

    // Validate directory exists.
    if (!is_dir($recipes_dir)) {
      $this->logger()->error("Recipes directory not found: $recipes_dir");
      return DrushCommands::EXIT_FAILURE;
    }

    // Find all recipe directories (containing recipe.yml).
    $recipe_dirs = [];
    $iterator = new \DirectoryIterator($recipes_dir);
    foreach ($iterator as $fileinfo) {
      if ($fileinfo->isDir() && !$fileinfo->isDot()) {
        $recipe_yml = $fileinfo->getPathname() . '/recipe.yml';
        if (file_exists($recipe_yml)) {
          $recipe_dirs[$fileinfo->getFilename()] = $fileinfo->getPathname();
        }
      }
    }

    if (empty($recipe_dirs)) {
      $this->logger()->warning("No recipes found in: $recipes_dir");
      return DrushCommands::EXIT_SUCCESS;
    }

    $total_recipes = count($recipe_dirs);
    $imported_count = 0;
    $failed_count = 0;

    $this->output()->writeln("\n<info>Found $total_recipes recipes to import</info>");
    $this->output()->writeln(str_repeat('=', 70));

    foreach ($recipe_dirs as $recipe_name => $recipe_path) {
      $this->output()->writeln("\n<comment>Importing: $recipe_name</comment>");
      $this->output()->writeln(str_repeat('-', 70));

      $result = $this->importRecipe($recipe_path, $options);

      if ($result === DrushCommands::EXIT_SUCCESS) {
        $imported_count++;
        $this->output()->writeln("<info>✓ Successfully imported $recipe_name</info>");
      }
      else {
        $failed_count++;
        $this->output()->writeln("<error>✗ Failed to import $recipe_name</error>");
      }
    }

    // Summary.
    $this->output()->writeln("\n" . str_repeat('=', 70));
    $this->output()->writeln("<info>Import All Summary:</info>");
    $this->output()->writeln("  Total recipes: $total_recipes");
    $this->output()->writeln("  <info>Successfully imported: $imported_count</info>");
    if ($failed_count > 0) {
      $this->output()->writeln("  <error>Failed: $failed_count</error>");
    }

    return $failed_count > 0 ? DrushCommands::EXIT_FAILURE : DrushCommands::EXIT_SUCCESS;
  }

  /**
   * Check if a bundle exists for an entity type.
   *
   * @param string $entity_type
   *   The entity type.
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return bool
   *   TRUE if the bundle exists, FALSE otherwise.
   */
  protected function bundleExists(string $entity_type, string $bundle): bool {
    $bundles = $this->getBundles($entity_type);
    return isset($bundles[$bundle]);
  }

  /**
   * Get all bundles for an entity type.
   *
   * @param string $entity_type
   *   The entity type.
   *
   * @return array
   *   Array of bundle IDs and labels.
   */
  protected function getBundles(string $entity_type): array {
    $bundles = [];

    try {
      // Special handling for different entity types.
      if ($entity_type === 'paragraph') {
        $storage = $this->entityTypeManager->getStorage('paragraphs_type');
      }
      elseif ($entity_type === 'taxonomy_term') {
        $storage = $this->entityTypeManager->getStorage('taxonomy_vocabulary');
      }
      else {
        $bundle_entity_type = $this->entityTypeManager->getDefinition($entity_type)->getBundleEntityType();
        if ($bundle_entity_type) {
          $storage = $this->entityTypeManager->getStorage($bundle_entity_type);
        }
        else {
          return $bundles;
        }
      }

      $entities = $storage->loadMultiple();
      foreach ($entities as $id => $entity) {
        $bundles[$id] = $entity->label();
      }
    }
    catch (\Exception $e) {
      $this->logger()->warning("Could not load bundles for $entity_type: " . $e->getMessage());
    }

    return $bundles;
  }

  /**
   * Get base modules required for an entity type.
   *
   * @param string $entity_type
   *   The entity type.
   *
   * @return array
   *   Array of module names.
   */
  protected function getBaseModules(string $entity_type): array {
    $base_modules = ['field', 'text'];

    $module_map = [
      'node' => ['node'],
      'paragraph' => ['paragraphs'],
      'media' => ['media'],
      'taxonomy_term' => ['taxonomy'],
      'block_content' => ['block_content'],
    ];

    if (isset($module_map[$entity_type])) {
      $base_modules = array_merge($base_modules, $module_map[$entity_type]);
    }

    return $base_modules;
  }

  /**
   * Export a configuration object.
   *
   * @param string $config_name
   *   The configuration name.
   * @param string $config_dir
   *   The destination directory.
   * @param array $exported_configs
   *   Array to track exported configs.
   */
  protected function exportConfig(string $config_name, string $config_dir, array &$exported_configs) {
    $config_data = $this->configStorage->read($config_name);

    if ($config_data === FALSE) {
      $this->logger()->debug("Configuration '$config_name' not found (may be optional).");
      return;
    }

    $filename = $config_dir . '/' . $config_name . '.yml';
    $yaml_content = Yaml::dump($config_data, 6, 2);

    $this->filesystem->dumpFile($filename, $yaml_content);
    $exported_configs[] = $config_name;

    $this->output()->writeln("  Exported: $config_name");
  }

  /**
   * Analyze field dependencies for media and taxonomy references.
   *
   * @param array $config_data
   *   The field configuration data.
   * @param array $dependencies
   *   The dependencies array to update.
   */
  protected function analyzeDependencies(array $config_data, array &$dependencies) {
    // Check for entity reference fields.
    if (isset($config_data['field_type']) && $config_data['field_type'] === 'entity_reference') {
      $target_type = NULL;

      // Try to get target_type from settings (field storage).
      if (isset($config_data['settings']['target_type'])) {
        $target_type = $config_data['settings']['target_type'];
      }
      // Try to parse target_type from handler (field instance).
      elseif (isset($config_data['settings']['handler'])) {
        // Handler format: 'default:taxonomy_term' or 'default:media'.
        $handler_parts = explode(':', $config_data['settings']['handler']);
        if (count($handler_parts) === 2) {
          $target_type = $handler_parts[1];
        }
      }

      if ($target_type) {
        if ($target_type === 'media') {
          if (!in_array('media', $dependencies['modules'])) {
            $dependencies['modules'][] = 'media';
          }
          if (isset($config_data['settings']['handler_settings']['target_bundles'])) {
            foreach ($config_data['settings']['handler_settings']['target_bundles'] as $bundle) {
              if (!in_array($bundle, $dependencies['media_types'])) {
                $dependencies['media_types'][] = $bundle;
              }
            }
          }
        }
        elseif ($target_type === 'taxonomy_term') {
          if (!in_array('taxonomy', $dependencies['modules'])) {
            $dependencies['modules'][] = 'taxonomy';
          }
          if (isset($config_data['settings']['handler_settings']['target_bundles'])) {
            foreach ($config_data['settings']['handler_settings']['target_bundles'] as $bundle) {
              if (!in_array($bundle, $dependencies['vocabularies'])) {
                $dependencies['vocabularies'][] = $bundle;
              }
            }
          }
        }
      }
    }

    // Check for image/file fields.
    if (isset($config_data['field_type'])) {
      if ($config_data['field_type'] === 'image' && !in_array('image', $dependencies['modules'])) {
        $dependencies['modules'][] = 'image';
      }
      elseif ($config_data['field_type'] === 'file' && !in_array('file', $dependencies['modules'])) {
        $dependencies['modules'][] = 'file';
      }
    }
  }

  /**
   * Export referenced entity type configurations.
   *
   * @param array $dependencies
   *   The dependencies array.
   * @param string $config_dir
   *   The destination directory.
   * @param array $exported_configs
   *   Array to track exported configs.
   */
  protected function exportReferencedEntities(array $dependencies, string $config_dir, array &$exported_configs) {
    // Export media types.
    foreach ($dependencies['media_types'] as $media_type) {
      $this->output()->writeln("\n<comment>Exporting referenced media type: $media_type</comment>");
      $this->exportConfig("media.type.$media_type", $config_dir, $exported_configs);

      $media_fields = $this->configStorage->listAll("field.field.media.$media_type");
      foreach ($media_fields as $config_name) {
        $this->exportConfig($config_name, $config_dir, $exported_configs);
      }

      $media_form_displays = $this->configStorage->listAll("core.entity_form_display.media.$media_type");
      foreach ($media_form_displays as $config_name) {
        $this->exportConfig($config_name, $config_dir, $exported_configs);
      }

      $media_view_displays = $this->configStorage->listAll("core.entity_view_display.media.$media_type");
      foreach ($media_view_displays as $config_name) {
        $this->exportConfig($config_name, $config_dir, $exported_configs);
      }
    }

    // Export taxonomy vocabularies.
    foreach ($dependencies['vocabularies'] as $vocab) {
      $this->output()->writeln("\n<comment>Exporting referenced taxonomy vocabulary: $vocab</comment>");
      $this->exportConfig("taxonomy.vocabulary.$vocab", $config_dir, $exported_configs);

      // Get all field instances for this vocabulary to determine which storage is needed.
      $vocab_field_configs = $this->configStorage->listAll("field.field.taxonomy_term.$vocab");
      $vocab_used_storages = [];

      foreach ($vocab_field_configs as $config_name) {
        $this->exportConfig($config_name, $config_dir, $exported_configs);

        // Track which field storage this field uses.
        $config_data = $this->configStorage->read($config_name);
        if (isset($config_data['field_name'])) {
          $storage_name = "field.storage.taxonomy_term." . $config_data['field_name'];
          $vocab_used_storages[$storage_name] = TRUE;
        }
      }

      // Export only the field storage configurations that are actually used by this vocabulary.
      foreach (array_keys($vocab_used_storages) as $storage_config_name) {
        $this->exportConfig($storage_config_name, $config_dir, $exported_configs);
      }

      // Export form displays.
      $vocab_form_displays = $this->configStorage->listAll("core.entity_form_display.taxonomy_term.$vocab");
      foreach ($vocab_form_displays as $config_name) {
        $this->exportConfig($config_name, $config_dir, $exported_configs);
      }

      // Export view displays.
      $vocab_view_displays = $this->configStorage->listAll("core.entity_view_display.taxonomy_term.$vocab");
      foreach ($vocab_view_displays as $config_name) {
        $this->exportConfig($config_name, $config_dir, $exported_configs);
      }
    }
  }

  /**
   * Export configs that reference the bundle (field_validation, etc).
   *
   * This method scans ALL configurations in the system and exports any that
   * reference the bundle being exported. This catches edge cases like:
   * - field_validation.rule_set configs
   * - conditional_fields configs
   * - Any other contrib module configs that reference bundles
   *
   * Uses bundle-level detection only to avoid false positives from Views,
   * Search API indexes, and other configs that may reference field names.
   *
   * @param string $entity_type
   *   The entity type.
   * @param string $bundle
   *   The bundle machine name.
   * @param string $config_dir
   *   The destination directory.
   * @param array $exported_configs
   *   Array to track exported configs.
   */
  protected function exportBundleReferencingConfigs(string $entity_type, string $bundle, string $config_dir, array &$exported_configs) {
    $this->output()->writeln("\n<comment>Scanning for configs that reference this bundle...</comment>");

    // Get all config names in the system.
    $all_config_names = $this->configStorage->listAll();
    $found_count = 0;

    foreach ($all_config_names as $config_name) {
      // Skip configs we've already exported.
      if (in_array($config_name, $exported_configs)) {
        continue;
      }

      // Skip core config prefixes that we know don't reference bundles.
      $skip_prefixes = [
        'core.entity_form_display.',
        'core.entity_view_display.',
        'field.field.',
        'field.storage.',
        'language.content_settings.',
        'system.',
        'user.role.',
      ];

      $should_skip = FALSE;
      foreach ($skip_prefixes as $prefix) {
        if (strpos($config_name, $prefix) === 0) {
          $should_skip = TRUE;
          break;
        }
      }

      if ($should_skip) {
        continue;
      }

      // Read the config and check if it references our bundle.
      $config_data = $this->configStorage->read($config_name);
      if ($config_data === FALSE) {
        continue;
      }

      // Check if this config references our entity_type and bundle.
      $references_bundle = FALSE;

      // Method 1: Check for entity_type and bundle keys.
      if (isset($config_data['entity_type']) && $config_data['entity_type'] === $entity_type &&
          isset($config_data['bundle']) && $config_data['bundle'] === $bundle) {
        $references_bundle = TRUE;
      }

      // Method 2: Check for targetEntityType and bundle keys (some modules use this).
      if (isset($config_data['targetEntityType']) && $config_data['targetEntityType'] === $entity_type &&
          isset($config_data['bundle']) && $config_data['bundle'] === $bundle) {
        $references_bundle = TRUE;
      }

      // Method 3: Check naming pattern like <module>.<type>.<entity_type>_<bundle>.
      $pattern = preg_quote($entity_type . '_' . $bundle, '/');
      if (preg_match('/' . $pattern . '/', $config_name)) {
        // Double-check the config actually references our bundle.
        $config_string = serialize($config_data);
        if (strpos($config_string, $entity_type) !== FALSE && strpos($config_string, $bundle) !== FALSE) {
          $references_bundle = TRUE;
        }
      }

      if ($references_bundle) {
        $this->exportConfig($config_name, $config_dir, $exported_configs);
        $found_count++;
      }
    }

    if ($found_count > 0) {
      $this->output()->writeln("<info>Found and exported $found_count bundle-referencing configs</info>");
    }
    else {
      $this->output()->writeln("<comment>No additional bundle-referencing configs found</comment>");
    }
  }

  /**
   * Scan all exported config files for module dependencies.
   *
   * @param string $config_dir
   *   The configuration directory.
   * @param array $dependencies
   *   The dependencies array to update.
   */
  protected function scanConfigDependencies(string $config_dir, array &$dependencies) {
    $config_files = glob($config_dir . '/*.yml');

    foreach ($config_files as $config_file) {
      $config_data = Yaml::parseFile($config_file);
      $config_name = basename($config_file, '.yml');

      // Check for module dependencies in the config.
      if (isset($config_data['dependencies']['module'])) {
        foreach ($config_data['dependencies']['module'] as $module) {
          if (!in_array($module, $dependencies['modules'])) {
            $dependencies['modules'][] = $module;
            $this->output()->writeln("  Found module dependency: $module");
          }
        }
      }

      // Infer module from config name prefix (e.g., field_validation.* requires field_validation module).
      // Skip core prefixes.
      $core_prefixes = ['core.', 'field.', 'language.', 'node.', 'paragraph.', 'media.', 'taxonomy.', 'block_content.', 'system.', 'user.'];
      $is_core = FALSE;
      foreach ($core_prefixes as $prefix) {
        if (strpos($config_name, $prefix) === 0) {
          $is_core = TRUE;
          break;
        }
      }

      if (!$is_core) {
        // Extract potential module name from config prefix.
        $parts = explode('.', $config_name);
        if (count($parts) >= 2) {
          $potential_module = $parts[0];
          // Check if this module exists and isn't already in dependencies.
          if (\Drupal::moduleHandler()->moduleExists($potential_module) && !in_array($potential_module, $dependencies['modules'])) {
            $dependencies['modules'][] = $potential_module;
            $this->output()->writeln("  Found module dependency from config prefix: $potential_module");
          }
        }
      }
    }
  }

  /**
   * Create recipe.yml file.
   *
   * @param string $entity_type
   *   The entity type.
   * @param string $bundle
   *   The bundle machine name.
   * @param string $destination
   *   The destination directory.
   * @param array $dependencies
   *   The dependencies array.
   */
  protected function createRecipeYml(string $entity_type, string $bundle, string $destination, array $dependencies) {
    $modules = array_values(array_unique($dependencies['modules']));

    $recipe_data = [
      'name' => ucfirst($entity_type) . ' Bundle: ' . $bundle,
      'description' => "Exports the $bundle bundle for $entity_type entity type with all dependencies",
      'type' => 'Content type',
      'install' => $modules,
      'config' => [
        'import' => [
          'config' => '*',
        ],
      ],
    ];

    $yaml_content = Yaml::dump($recipe_data, 4, 2);
    $this->filesystem->dumpFile($destination . '/recipe.yml', $yaml_content);

    $this->output()->writeln("\n<info>Created recipe.yml with " . count($modules) . " modules</info>");
  }

  /**
   * Create README.md file.
   *
   * @param string $entity_type
   *   The entity type.
   * @param string $bundle
   *   The bundle machine name.
   * @param string $destination
   *   The destination directory.
   * @param array $exported_configs
   *   The exported configurations.
   * @param array $dependencies
   *   The dependencies array.
   */
  protected function createReadme(string $entity_type, string $bundle, string $destination, array $exported_configs, array $dependencies) {
    $readme = "# " . ucfirst($entity_type) . " Bundle Export: $bundle\n\n";
    $readme .= "This recipe exports the **$bundle** bundle for **$entity_type** entity type with all its dependencies.\n\n";
    $readme .= "## What's Included\n\n";
    $readme .= "- Bundle/type definition\n";
    $readme .= "- All field configurations (" . count(array_filter($exported_configs, fn($c) => strpos($c, 'field.') === 0)) . " configs)\n";
    $readme .= "- Form and view display configurations\n";

    if (!empty($dependencies['media_types'])) {
      $readme .= "- Referenced media types: " . implode(', ', $dependencies['media_types']) . "\n";
    }

    if (!empty($dependencies['vocabularies'])) {
      $readme .= "- Referenced taxonomy vocabularies: " . implode(', ', $dependencies['vocabularies']) . "\n";
    }

    $readme .= "\n## Installation\n\n";
    $readme .= "### On the target Drupal site:\n\n";
    $readme .= "1. Copy this directory to the target site\n\n";
    $readme .= "2. Apply the recipe:\n";
    $readme .= "   ```bash\n";
    $readme .= "   php docroot/core/scripts/drupal recipe " . basename($destination) . "\n";
    $readme .= "   ```\n\n";
    $readme .= "## Exported Configurations\n\n";
    $readme .= "Total: " . count($exported_configs) . " configuration files\n\n";

    $readme .= "## Dependencies\n\n";
    $readme .= "Required modules:\n";
    foreach ($dependencies['modules'] as $module) {
      $readme .= "- $module\n";
    }

    $readme .= "\n## Export Details\n\n";
    $readme .= "- Entity Type: $entity_type\n";
    $readme .= "- Bundle: $bundle\n";
    $readme .= "- Export Date: " . date('Y-m-d H:i:s') . "\n";

    $this->filesystem->dumpFile($destination . '/README.md', $readme);
    $this->output()->writeln("<info>Created README.md</info>");
  }

}
