<?php

declare(strict_types=1);

namespace Drupal\recipes_ui\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\recipes_ui\RecipeManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\File\FileSystemInterface;

/**
 * Form for listing and applying recipes.
 */
class RecipeListForm extends FormBase {

  /**
   * The recipe manager service.
   *
   * @var \Drupal\recipes_ui\RecipeManager
   */
  protected RecipeManager $recipeManager;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * Constructs a RecipeListForm object.
   *
   * @param \Drupal\recipes_ui\RecipeManager $recipe_manager
   *   The recipe manager service.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   */
  public function __construct(RecipeManager $recipe_manager, FileSystemInterface $file_system) {
    $this->recipeManager = $recipe_manager;
    $this->fileSystem = $file_system;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('recipes_ui.recipe_manager'),
      $container->get('file_system')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'recipes_ui_list_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $recipes = $this->recipeManager->discoverRecipes();

    if (empty($recipes)) {
      $form['no_recipes'] = [
        '#markup' => '<p>' . $this->t('No recipes found. Place recipe folders in the ../recipes directory (outside the webroot). Or in the core/recipes directory.') . '</p>',
      ];
      return $form;
    }

    $form['description'] = [
      '#markup' => '<p>' . $this->t('Select one or more recipes to apply. Recipes will install modules, themes, configuration, and content as defined in their recipe.yml files.') . '</p>',
    ];

    $form['recipes'] = [
      '#type' => 'tableselect',
      '#header' => [
        'name' => $this->t('Name'),
        'description' => $this->t('Description'),
        'type' => $this->t('Type'),
        'path' => $this->t('Path'),
      ],
      '#options' => [],
      '#empty' => $this->t('No recipes available.'),
      '#multiple' => TRUE,
    ];

    foreach ($recipes as $recipe_path => $recipe_info) {
      // Convert absolute path to relative path from Drupal root.
      $drupal_root = $this->fileSystem->realpath('');
      $relative_path = $recipe_path;
      if (strpos($recipe_path, $drupal_root) === 0) {
        $relative_path = '.' . substr($recipe_path, strlen($drupal_root));
      }

      $form['recipes']['#options'][$recipe_path] = [
        'name' => $recipe_info['name'],
        'description' => $recipe_info['description'] ?: $this->t('No description'),
        'type' => $recipe_info['type'] ?: $this->t('Standard'),
        'path' => $relative_path,
      ];
    }

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['apply'] = [
      '#type' => 'submit',
      '#value' => $this->t('Apply Selected Recipes'),
      '#button_type' => 'primary',
    ];

    $form['actions']['apply_batch'] = [
      '#type' => 'submit',
      '#value' => $this->t('Apply Selected Recipes (Batch)'),
      '#submit' => ['::submitFormBatch'],
    ];

    $form['actions']['refresh'] = [
      '#type' => 'submit',
      '#value' => $this->t('Refresh Recipe List'),
      '#submit' => ['::submitFormRefresh'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $recipes_value = $form_state->getValue('recipes');

    // Ensure we have an array to work with.
    if (!is_array($recipes_value)) {
      $recipes_value = [];
    }

    $selected_recipes = array_filter($recipes_value);

    if (empty($selected_recipes)) {
      $form_state->setErrorByName('recipes', $this->t('Please select at least one recipe to apply.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $recipes_value = $form_state->getValue('recipes');

    // Ensure we have an array to work with.
    if (!is_array($recipes_value)) {
      $recipes_value = [];
    }

    $selected_recipes = array_filter($recipes_value);

    if (empty($selected_recipes)) {
      return;
    }

    $recipes = $this->recipeManager->discoverRecipes();
    $success_count = 0;
    $error_count = 0;
    $applied_recipes = [];
    $failed_recipes = [];

    foreach ($selected_recipes as $recipe_path => $selected) {
      if ($selected) {
        $recipe_name = $recipes[$recipe_path]['name'] ?? basename($recipe_path);
        $success = $this->recipeManager->applyRecipe($recipe_path);

        if ($success) {
          $success_count++;
          $applied_recipes[] = $recipe_name;
        }
        else {
          $error_count++;
          $failed_recipes[] = $recipe_name;
        }
      }
    }

    // Display results.
    if ($success_count > 0) {
      $this->messenger()->addStatus($this->t('Successfully applied @count recipe(s): @recipes', [
        '@count' => $success_count,
        '@recipes' => implode(', ', $applied_recipes),
      ]));
    }

    if ($error_count > 0) {
      $this->messenger()->addError($this->t('Failed to apply @count recipe(s): @recipes. Check the logs for more details.', [
        '@count' => $error_count,
        '@recipes' => implode(', ', $failed_recipes),
      ]));
    }
  }

  /**
   * Submit handler for batch application.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitFormBatch(array &$form, FormStateInterface $form_state): void {
    $recipes_value = $form_state->getValue('recipes');

    // Ensure we have an array to work with.
    if (!is_array($recipes_value)) {
      $recipes_value = [];
    }

    $selected_recipes = array_filter($recipes_value);

    if (empty($selected_recipes)) {
      return;
    }

    $recipes = $this->recipeManager->discoverRecipes();
    $success_count = 0;
    $error_count = 0;
    $started_recipes = [];
    $failed_recipes = [];

    foreach ($selected_recipes as $recipe_path => $selected) {
      if ($selected) {
        $recipe_name = $recipes[$recipe_path]['name'] ?? basename($recipe_path);
        $success = $this->recipeManager->applyRecipeBatch($recipe_path);

        if ($success) {
          $success_count++;
          $started_recipes[] = $recipe_name;
        }
        else {
          $error_count++;
          $failed_recipes[] = $recipe_name;
        }
      }
    }

    // Display results.
    if ($success_count > 0) {
      $this->messenger()->addStatus($this->t('Batch process started for @count recipe(s): @recipes', [
        '@count' => $success_count,
        '@recipes' => implode(', ', $started_recipes),
      ]));
    }

    if ($error_count > 0) {
      $this->messenger()->addError($this->t('Failed to start batch process for @count recipe(s): @recipes. Check the logs for more details.', [
        '@count' => $error_count,
        '@recipes' => implode(', ', $failed_recipes),
      ]));
    }
  }

  /**
   * Submit handler for refreshing the recipe list.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitFormRefresh(array &$form, FormStateInterface $form_state): void {
    $this->messenger()->addStatus($this->t('Recipe list refreshed.'));
  }

}
