<?php

declare(strict_types=1);

namespace Drupal\recipes_ui;

use Drupal\Core\Recipe\RecipeRunner;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Recipe\Recipe;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;

/**
 * Service for managing and discovering recipes.
 */
class RecipeManager {

  use StringTranslationTrait;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * The logger factory service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $loggerFactory;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected MessengerInterface $messenger;

  /**
   * Constructs a RecipeManager object.
   *
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(FileSystemInterface $file_system, TranslationInterface $string_translation, LoggerChannelFactoryInterface $logger_factory, MessengerInterface $messenger) {
    $this->fileSystem = $file_system;
    $this->stringTranslation = $string_translation;
    $this->loggerFactory = $logger_factory;
    $this->messenger = $messenger;
  }

  /**
   * Discovers all available recipes in the system.
   *
   * @return array
   *   An array of recipe information, keyed by recipe path.
   */
  public function discoverRecipes(): array {
    $recipes = [];

    // Search in the recipes directory outside the webroot.
    $recipes_dir = $this->fileSystem->realpath('') . '/../recipes';
    if (is_dir($recipes_dir)) {
      $recipes = array_merge($recipes, $this->scanDirectoryForRecipes($recipes_dir));
    }

    // Search in core recipes directory.
    $core_recipes_dir = $this->fileSystem->realpath('') . '/core/recipes';
    if (is_dir($core_recipes_dir)) {
      $recipes = array_merge($recipes, $this->scanDirectoryForRecipes($core_recipes_dir));
    }

    return $recipes;
  }

  /**
   * Scans a directory for recipe folders.
   *
   * @param string $directory
   *   The directory to scan.
   *
   * @return array
   *   An array of recipe information.
   */
  protected function scanDirectoryForRecipes(string $directory): array {
    $recipes = [];

    if (!is_dir($directory)) {
      return $recipes;
    }

    $items = scandir($directory);
    foreach ($items as $item) {
      if ($item === '.' || $item === '..') {
        continue;
      }

      $recipe_path = $directory . '/' . $item;
      if (is_dir($recipe_path) && file_exists($recipe_path . '/recipe.yml')) {
        try {
          $recipe = Recipe::createFromDirectory($recipe_path);
          $recipes[$recipe_path] = [
            'path' => $recipe_path,
            'name' => $recipe->name,
            'description' => $recipe->description,
            'type' => $recipe->type,
            'recipe_object' => $recipe,
          ];
        }
        catch (\Exception $e) {
          // Skip invalid recipes.
          continue;
        }
      }
    }

    return $recipes;
  }

  /**
   * Applies a recipe by its path.
   *
   * @param string $recipe_path
   *   The path to the recipe directory.
   *
   * @return bool
   *   TRUE if the recipe was applied successfully, FALSE otherwise.
   */
  public function applyRecipe(string $recipe_path): bool {
    try {
      $recipe = Recipe::createFromDirectory($recipe_path);
      RecipeRunner::processRecipe($recipe);
      return TRUE;
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('recipes_ui')->error('Failed to apply recipe @path: @message', [
        '@path' => $recipe_path,
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Applies a recipe using batch operations.
   *
   * @param string $recipe_path
   *   The path to the recipe directory.
   *
   * @return bool
   *   TRUE if the batch was started successfully, FALSE otherwise.
   */
  public function applyRecipeBatch(string $recipe_path): bool {
    try {
      $recipe = Recipe::createFromDirectory($recipe_path);
      $batch_operations = RecipeRunner::toBatchOperations($recipe);

      if (empty($batch_operations)) {
        return FALSE;
      }

      $batch = [
        'title' => $this->t('Applying recipe: @name', ['@name' => $recipe->name]),
        'operations' => $batch_operations,
        'finished' => [static::class, 'batchFinished'],
      ];

      batch_set($batch);
      return TRUE;
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('recipes_ui')->error('Failed to start batch for recipe @path: @message', [
        '@path' => $recipe_path,
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Batch finished callback.
   *
   * @param bool $success
   *   Whether the batch completed successfully.
   * @param array $results
   *   The batch results.
   * @param array $operations
   *   The batch operations.
   */
  public static function batchFinished(bool $success, array $results, array $operations): void {
    $messenger = \Drupal::service('messenger');
    $translation = \Drupal::service('string_translation');
    if ($success) {
      $messenger->addStatus($translation->translate('Recipe applied successfully.'));
    }
    else {
      $messenger->addError($translation->translate('Recipe application failed.'));
    }
  }

}
