<?php

namespace Drupal\recurlyjs\Form;

use Drupal\Component\Utility\Html;
use Drupal\Core\Form\FormStateInterface;
use Drupal\recurlyjs\Event\SubscriptionAlter;
use Drupal\recurlyjs\Event\SubscriptionCreated;
use Drupal\recurlyjs\RecurlyJsEvents;
use Recurly\Errors\NotFound;
use Recurly\Errors\Validation;
use Recurly\RecurlyError;
use Recurly\Resources\Coupon;

/**
 * RecurlyJS subscribe form.
 */
class RecurlyJsSubscribeForm extends RecurlyJsFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'recurlyjs_subscribe';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $entity_type = NULL, $entity = NULL, $plan_code = NULL, $currency = NULL) {
    if (!$entity_type || !$entity || !$plan_code) {
      // One or more required parameters are missing, so show a user-friendly
      // error and prevent the form from being submitted.
      $this->messenger()->addError($this->t('Unable to load the subscription form because required information is missing.'));
      $form_state->setErrorByName('plan_code', $this->t('The subscription you are trying to access could not be found.'));

      $form['message'] = [
        '#markup' => $this->t('The subscription you are trying to access could not be found.'),
      ];

      return $form;
    }

    $form = parent::buildForm($form, $form_state);
    $form['#entity_type'] = $entity_type;
    $form['#entity'] = $entity;
    $form['#plan_code'] = $plan_code;
    $form['#currency'] = $currency ?: $this->config('recurly.settings')->get('recurly_default_currency') ?: 'USD';

    // Display a summary of what the user is about to purchase.
    $currency = $this->config('recurly.settings')->get('recurly_default_currency');
    /** @var \Recurly\Resources\Plan $plan */
    $plan = $this->recurlyClient->getPlan('code-' . $plan_code);
    $unit_amount = NULL;
    foreach ($plan->getCurrencies() as $unit_currency) {
      if ($unit_currency->getCurrency() === $currency) {
        $unit_amount = $this->formatManager->formatCurrency($unit_currency->getUnitAmount(), $unit_currency->getCurrency(), TRUE);
        break;
      }
    }
    $frequency = $this->formatManager->formatPriceInterval($unit_amount, $plan->getIntervalLength(), $plan->getIntervalUnit(), FALSE);

    $form['plan'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Plan'),
      '#weight' => -350,
    ];
    $form['plan']['plan_name'] = [
      '#markup' => '<div class="recurlyjs-element recurlyjs-plan-name">' . HTML::escape($plan->getName()) . ' <span class="recurlyjs-plan-frequency">(' . $frequency . ')</span></div>',
    ];
    $description = $plan->getDescription();
    if ($description) {
      $form['plan']['plan_description'] = [
        '#markup' => '<div class="recurlyjs-element recurlyjs-plan-description">' . HTML::escape($description) . '</span></div>',
      ];
    }
    $form['plan']['addons'] = [
      '#markup' => '<h2 id="addons-title" class="recurlyjs-element__hidden">Add-ons</h2><div id="addons"></div>',
    ];

    // Use the recurly.js recurly.Pricing() module to display a summary of what
    // the user is going to purchase. For more information about
    // recurly.Pricing() see https://dev.recurly.com/docs/recurly-js-pricing.
    $form['plan']['pricing'] = [
      '#type' => '#container',
      '#attributes' => [
        'class' => ['recurlyjs-pricing'],
      ],
    ];
    // For the recurly.Pricing() module, we need to make sure we include an
    // input field with the plan name.
    $form['plan']['pricing']['plan_code'] = [
      '#type' => 'hidden',
      '#value' => $plan_code,
      '#attributes' => [
        'data-recurly' => 'plan',
      ],
    ];

    // Setup fee. Hidden by default, populated by JS as needed.
    $form['plan']['pricing']['plan_setup'] = [
      '#markup' => '<div class="recurlyjs-element recurlyjs-setup-fee recurlyjs-element__hidden">' . $this->t('Setup fee:') . ' <span data-recurly="currency_symbol"></span><span data-recurly="setup_fee_now"></span></div>',
    ];

    // Discount. Hidden by default, populated by JS as needed.
    $form['plan']['pricing']['plan_discount'] = [
      '#markup' => '<div class="recurlyjs-element recurlyjs-discount recurlyjs-element__hidden">' . $this->t('Discount:') . ' <span data-recurly="currency_symbol"></span><span data-recurly="discount_now"></span></div>',
    ];

    // Sub total. Hidden by default, populated by JS as needed.
    $form['plan']['pricing']['plan_subtotal'] = [
      '#markup' => '<div class="recurlyjs-element recurlyjs-subtotal recurlyjs-element__hidden">' . $this->t('Subtotal:') . ' <span data-recurly="currency_symbol"></span><span data-recurly="subtotal_now"></span></div>',
    ];

    // Taxes. Hidden by default, populated by JS as needed.
    $form['plan']['pricing']['plan_tax'] = [
      '#markup' => '<div class="recurlyjs-element recurlyjs-tax recurlyjs-element__hidden">' . $this->t('Taxes:') . ' <span data-recurly="currency_symbol"></span><span data-recurly="tax_now"></span></div>',
    ];

    // Sub total. Hidden by default, populated by JS as needed.
    $form['plan']['pricing']['plan_total'] = [
      '#markup' => '<div class="recurlyjs-element recurlyjs-total recurlyjs-element__hidden">' . $this->t('Total due now:') . ' <span data-recurly="currency_symbol"></span><span data-recurly="total_now"></span></div>',
    ];

    if ($this->config('recurlyjs.settings')->get('recurlyjs_enable_quantity') || $this->config('recurlyjs.settings')->get('recurlyjs_enable_coupons')) {
      $form['billing_extras'] = [
        '#type' => 'fieldset',
      ];
    }

    // Controlled by a setting in the base recurly module.
    if ($this->config('recurly.settings')->get('recurly_subscription_multiple')) {
      $form['billing_extras']['quantity'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Quantity'),
        '#description' => $this->t('Number of subscriptions to this plan.'),
        '#size' => 3,
        '#default_value' => $this->getRequest()->query->get('quantity', 1),
        '#attributes' => [
          'data-recurly' => 'plan_quantity',
        ],
        '#weight' => '-300',
      ];
    }

    if ($this->config('recurlyjs.settings')->get('recurlyjs_enable_coupons')) {
      $form['billing_extras']['coupon_code'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Coupon Code'),
        '#description' => $this->t('Recurly coupon code to be applied to subscription.'),
        '#element_validate' => ['::validateCouponCode'],
        '#attributes' => [
          'data-recurly' => 'coupon',
        ],
        '#weight' => -50,
      ];
    }

    $form['actions'] = [
      '#type' => 'actions',
    ];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Purchase'),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $entity_type = $form['#entity_type'];
    $entity = $form['#entity'];
    $plan_code = $form['#plan_code'];
    $currency = $form['#currency'];
    $recurly_token = $form_state->getValue('recurly-token');
    $coupon_code = $form_state->getValue('coupon_code');
    $quantity = $form_state->getValue('quantity') ? $form_state->getValue('quantity') : 1;
    $recurly_account = recurly_account_load([
      'entity_type' => $entity_type,
      'entity_id' => $entity->id(),
    ]);

    $subscription = [];
    if (!$recurly_account) {
      $subscription['account'] = [
        'first_name' => Html::escape($form_state->getValue('first_name')),
        'last_name' => Html::escape($form_state->getValue('last_name')),
      ];

      if ($entity_type == 'user') {
        $subscription['account']['email'] = $entity->getEmail();
        $subscription['account']['username'] = $entity->getAccountName();
      }

      // Use token mapping for new recurly account fields.
      foreach ($this->tokenManager->tokenMapping() as $recurly_field => $token) {
        $token_value = $this->token->replace(
          $token,
          [$entity_type => $entity],
          ['clear' => TRUE, 'sanitize' => FALSE]
        );

        if (!empty($token_value)) {
          $this->tokenManager->mapTokenFieldsToRecurlyStructure($recurly_field, $token_value, $subscription['account']);
        }
      }

      // Account code is the only property required for account creation.
      $subscription['account']['code'] = $entity_type . '-' . $entity->id();
    }
    else {
      $subscription['account'] = [
        'code' => $recurly_account->getCode(),
      ];
    }

    // Add the subscription details to the purchase.
    $subscription['currency'] = $currency;
    $subscription['plan_code'] = $plan_code;
    $subscription['quantity'] = $quantity;
    if ($coupon_code) {
      $subscription['coupon_codes'] = [$coupon_code];
    }

    // Allow other modules the chance to alter the new Recurly Subscription
    // object before it is saved.
    $event = new SubscriptionAlter($subscription, $entity);
    $this->eventDispatcher->dispatch($event, RecurlyJsEvents::SUBSCRIPTION_ALTER);
    $subscription = $event->getSubscription();

    // Billing info is based on the token we retrieved from the Recurly JS API
    // and should only contain the token in this case. We add this after the
    // above alter hook to ensure it's not modified.
    $subscription['account']['billing_info'] = ['token_id' => $recurly_token];

    try {
      // This saves all of the data assembled above in addition to creating a
      // new subscription record.
      $recurly_subscription = $this->recurlyClient->createSubscription($subscription);
    }
    catch (Validation $e) {
      // There was an error validating information in the form. For example,
      // credit card was declined. We don't need to log these in Drupal, you can
      // find the errors logged within Recurly.
      $this->messenger()->addError($this->t('<strong>Unable to create subscription:</strong><br/>@error', ['@error' => $e->getMessage()]));
      $form_state->setRebuild(TRUE);
      return;
    }
    catch (RecurlyError $e) {
      // Catch any non-validation errors. This will be things like unable to
      // contact Recurly API, or lower level errors. Display a generic message
      // to the user letting them know there was an error and then log the
      // detailed version. There's probably nothing a user can do to correct
      // these errors so we don't need to display the details.
      $this->logger('recurlyjs')->error('Unable to create subscription. Received the following error: @error', ['@error' => $e->getMessage()]);
      $this->messenger()->addError($this->t('Unable to create subscription.'));
      $form_state->setRebuild(TRUE);
      return;
    }

    // Allow other modules to react to the new subscription being created.
    $event = new SubscriptionCreated($recurly_subscription, $entity);
    $this->eventDispatcher->dispatch($event, RecurlyJsEvents::SUBSCRIPTION_CREATED);
    $subscription = $event->getSubscription();

    $this->messenger()->addMessage($this->t('Account upgraded to @plan!', ['@plan' => $recurly_subscription->getPlan()->getName()]));

    // Save the account locally immediately so that subscriber information may
    // be retrieved when the user is directed back to the /subscription tab.
    try {
      $account = $recurly_subscription->getAccount();
      recurly_account_save_local(['code' => $account->getCode(), 'status' => $subscription->getState()], $entity_type, $entity->id());
    }
    catch (RecurlyError $e) {
      $this->logger('recurlyjs')->error('New subscriber account could not be retreived from Recurly. Received the following error: @error', ['@error' => $e->getMessage()]);
    }
    return $form_state->setRedirect("entity.$entity_type.recurly_subscriptionlist", [
      $entity->getEntityType()->id() => $entity->id(),
    ]);
  }

  /**
   * Element validate callback.
   */
  public function validateCouponCode($element, &$form_state, $form) {
    $coupon_code = $form_state->hasValue('coupon_code') ? $form_state->getValue('coupon_code') : NULL;
    if (!$coupon_code) {
      return;
    }
    $currency = $form['#currency'];
    $plan_code = $form['#plan_code'];

    // Query Recurly to make sure this is a valid coupon code.
    try {
      $coupon = $this->recurlyClient->getCoupon('code-' . $coupon_code);
    }
    catch (NotFound $e) {
      $form_state->setError($element, $this->t('The coupon code you have entered is not valid.'));
      return;
    }

    // Check that the coupon is available in the specified currency.
    if (!in_array($coupon->getDiscount()->getType(), ['percent', 'free_trial'])) {
      $valid_currencies = $coupon->getDiscount()->getCurrencies();
      $valid = FALSE;
      foreach ($valid_currencies as $valid_currency) {
        if ($valid_currency->getCurrency() === $currency) {
          $valid = TRUE;
          break;
        }
      }

      if (!$valid) {
        $form_state->setErrorByName('coupon_currency', $this->t('The coupon code you have entered is not valid in @currency currency.', ['@currency' => $form_state->getValue('coupon_currency')]));
      }
    }

    // Check if the coupon is valid for the specified plan.
    if (!$this->couponValidForPlan($coupon, $plan_code)) {
      $form_state->setError($element, $this->t('The coupon code you have entered is not valid for the specified plan.'));
      return;
    }
  }

  /**
   * Validate Recurly coupon against a specified plan.
   *
   * @param \Recurly\Resources\Coupon $recurly_coupon
   *   A Recurly coupon object.
   * @param string $plan_code
   *   A Recurly plan code.
   *
   * @return BOOL
   *   TRUE if the coupon is valid for the specified plan, else FALSE.
   */
  protected function couponValidForPlan(Coupon $recurly_coupon, $plan_code) {
    if ($recurly_coupon->getAppliesToAllPlans() === TRUE) {
      return TRUE;
    }

    /** @var \Recurly\Resources\Plan $plan */
    foreach ($recurly_coupon->getPlans() as $plan) {
      if ($plan->getCode() === $plan_code) {
        return TRUE;
      }
    }

    return FALSE;
  }

}
