<?php

namespace Drupal\recurlyjs\Form;

use Drupal\Component\Utility\Html;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Recurly\Errors\NotFound;
use Recurly\Errors\UnprocessableEntity;
use Recurly\RecurlyError;

/**
 * RecurlyJS update billing form.
 */
class RecurlyJsUpdateBillingForm extends RecurlyJsFormBase {

  /**
   * Card type string to match against Recurly response.
   */
  const CARD_TYPE_AMEX = 'American Express';

  /**
   * American Express card number length.
   */
  const CARD_LENGTH_AMEX = 11;

  /**
   * Standard credit card length.
   */
  const CARD_LENGTH_OTHER = 12;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'recurlyjs_update_billing';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?RouteMatchInterface $route_match = NULL) {
    $entity_type = $this->config('recurly.settings')->get('recurly_entity_type');
    $entity = $route_match->getParameter($entity_type);

    // See if we have a local mapping of entity ID to Recurly account code.
    $recurly_account = recurly_account_load([
      'entity_type' => $entity_type,
      'entity_id' => $entity->id(),
    ]);

    try {
      /** @var \Recurly\Resources\BillingInfo $billing_info */
      $billing_info = $this->recurlyClient->getBillingInfo($recurly_account->getId());
      // Format expiration date.
      $exp_date = sprintf('%1$02d', Html::escape($billing_info->getPaymentMethod()->getExpMonth())) . '/' . Html::escape($billing_info->getPaymentMethod()->getExpYear());
      // Determine the correct number of masked card numbers depending on the
      // type of card.
      $mask_length = strcasecmp($billing_info->getPaymentMethod()->getCardType(), self::CARD_TYPE_AMEX) === 0 ? self::CARD_LENGTH_AMEX : self::CARD_LENGTH_OTHER;
      $form['existing'] = [
        '#theme' => 'recurly_credit_card_information',
        '#card_type' => Html::escape($billing_info->getPaymentMethod()->getCardType()),
        '#first_name' => Html::escape($billing_info->getFirstName()),
        '#last_name' => Html::escape($billing_info->getLastName()),
        '#exp_date' => $exp_date,
        '#last_four' => Html::escape($billing_info->getPaymentMethod()->getLastFour()),
        '#card_num_masked' => str_repeat('x', $mask_length) . Html::escape($billing_info->getPaymentMethod()->getLastFour()),
      ];
    }
    catch (NotFound $e) {
      $this->logger('recurlyjs')->notice('Unable to retrieve billing information. Received the following error: @error', ['@error' => $e->getMessage()]);
      $this->messenger()->addError($this->t('Unable to retrieve billing information.'));
      return $form;
    }

    $form['#entity_type'] = $entity_type;
    $form['#entity'] = $entity;

    $this->setBillingInfo($billing_info);
    $form = parent::buildForm($form, $form_state);

    $form['actions'] = [
      '#type' => 'actions',
    ];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Update'),
      '#submit' => ['::submitForm'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);
    $entity_type = $form['#entity_type'];
    $entity = $form['#entity'];
    $recurly_token = $form_state->getValue('recurly-token');

    $recurly_account = recurly_account_load([
      'entity_type' => $entity_type,
      'entity_id' => $entity->id(),
    ]);

    if ($recurly_token && $recurly_account) {
      try {
        $billing_info = [
          'token_id' => $recurly_token,
        ];
        $this->recurlyClient->updateBillingInfo($recurly_account->getId(), $billing_info);
        $this->messenger()->addStatus($this->t('Billing information updated.'));
      }
      catch (UnprocessableEntity $e) {
        // There was an error validating information in the form. For example,
        // credit card was declined. Let the user know. These errors are logged
        // in Recurly.
        $this->messenger()->addError($this->t('<strong>Unable to update account:</strong><br/>@error', ['@error' => $e->getMessage()]));
      }
      catch (NotFound $e) {
        $this->messenger()->addError($this->t('Could not find account or token is invalid or expired.'));
        $form_state->setRebuild(TRUE);
      }
      catch (RecurlyError $e) {
        // Catch all other errors. Log the details, and display a message for
        // the user.
        $this->logger('recurlyjs')->error('Billing information update error: @error', ['@error' => $e->getMessage()]);
        $this->messenger()->addMessage($this->t('An error occured while trying to update your account. Please contact a site administrator.'));
        $form_state->setRebuild(TRUE);
      }
    }
  }

}
