<?php

namespace Drupal\recurlyjs;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Url;

/**
 * Provides helper methods for building Recurly JS URLs.
 */
class RecurlyJsLinkManager {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructs a new link manager.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * Builds the signup URL for an entity.
   */
  public function getSignupUrl(EntityInterface $entity, string $plan_code, ?string $currency = NULL): ?Url {
    $entity_type_id = $this->getEntityTypeId();
    if (!$entity_type_id || $entity->getEntityTypeId() !== $entity_type_id) {
      return NULL;
    }

    $route_parameters = [
      $entity_type_id => $entity->id(),
      'plan_code' => $plan_code,
    ];
    $options = [];
    if ($currency = $this->getNonDefaultCurrency($currency)) {
      $options['query']['currency'] = $currency;
    }

    $route_name = $this->getSignupRouteName($entity_type_id);
    return $route_name ? Url::fromRoute($route_name, $route_parameters, $options) : NULL;
  }

  /**
   * Builds the billing URL for an entity.
   */
  public function getBillingUrl(EntityInterface $entity): ?Url {
    $entity_type_id = $this->getEntityTypeId();
    if (!$entity_type_id || $entity->getEntityTypeId() !== $entity_type_id) {
      return NULL;
    }

    $route_name = $this->getBillingRouteName($entity_type_id);
    return $route_name ? Url::fromRoute($route_name, [$entity_type_id => $entity->id()]) : NULL;
  }

  /**
   * Gets the signup route name for the configured entity type.
   */
  public function getSignupRouteName(?string $entity_type_id = NULL): ?string {
    $entity_type_id = $entity_type_id ?: $this->getEntityTypeId();
    return $entity_type_id ? "recurlyjs.$entity_type_id.signup" : NULL;
  }

  /**
   * Gets the billing route name for the configured entity type.
   */
  public function getBillingRouteName(?string $entity_type_id = NULL): ?string {
    $entity_type_id = $entity_type_id ?: $this->getEntityTypeId();
    return $entity_type_id ? "recurlyjs.$entity_type_id.billing" : NULL;
  }

  /**
   * Returns the configured Recurly entity type ID.
   */
  protected function getEntityTypeId(): ?string {
    return $this->configFactory->get('recurly.settings')->get('recurly_entity_type') ?: NULL;
  }

  /**
   * Filters the provided currency so defaults aren't appended to URLs.
   */
  protected function getNonDefaultCurrency(?string $currency): ?string {
    if (!$currency) {
      return NULL;
    }

    $default_currency = $this->configFactory->get('recurly.settings')->get('recurly_default_currency') ?: 'USD';
    return $currency === $default_currency ? NULL : $currency;
  }

}
