<?php

namespace Drupal\recurlyjs\Routing;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Routing\Route;
use Symfony\Component\Routing\RouteCollection;

/**
 * Builds dynamic Recurly JS routes based on configuration.
 */
class RecurlyJsRoutes implements ContainerInjectionInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new RecurlyJsRoutes instance.
   */
  public function __construct(ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager) {
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * Route callback for Recurly JS routes.
   */
  public function routes(): RouteCollection {
    $collection = new RouteCollection();

    $entity_type_id = $this->configFactory
      ->get('recurly.settings')
      ->get('recurly_entity_type');
    if (!$entity_type_id) {
      return $collection;
    }

    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id, FALSE);
    if (!$entity_type || !$entity_type->hasLinkTemplate('canonical')) {
      return $collection;
    }

    $canonical = $entity_type->getLinkTemplate('canonical');
    $route_options = [
      '_admin_route' => TRUE,
      '_recurly_entity_type_id' => $entity_type_id,
      'parameters' => [
        $entity_type_id => [
          'type' => 'entity:' . $entity_type_id,
        ],
      ],
    ];

    $signup_route = $this->buildSignupRoute($canonical, $entity_type_id, $route_options);
    $collection->add("recurlyjs.$entity_type_id.signup", $signup_route);

    $billing_route = $this->buildBillingRoute($canonical, $entity_type_id, $route_options);
    $collection->add("recurlyjs.$entity_type_id.billing", $billing_route);

    return $collection;
  }

  /**
   * Builds the signup route.
   */
  protected function buildSignupRoute(string $canonical, string $entity_type_id, array $options): Route {
    $signup_options = $options;
    $signup_options['parameters']['plan_code'] = ['type' => 'string'];

    return new Route(
      $canonical . '/subscription/signup/{plan_code}',
      [
        '_controller' => '\Drupal\recurlyjs\Controller\RecurlyJsSubscriptionSignupController::subscribe',
        '_title' => 'Subscription signup',
        'operation' => 'signup',
        'currency' => NULL,
      ],
      [
        '_entity_access' => "$entity_type_id.update",
        '_access_check_recurly_user' => 'TRUE',
        '_access_check_recurly_list' => 'TRUE',
        '_access_check_recurly_signup' => 'TRUE',
      ],
      $signup_options
    );
  }

  /**
   * Builds the billing route.
   */
  protected function buildBillingRoute(string $canonical, string $entity_type_id, array $options): Route {
    return new Route(
      $canonical . '/subscription/billing',
      [
        '_form' => '\Drupal\recurlyjs\Form\RecurlyJsUpdateBillingForm',
        '_title' => 'Subscription billing',
        'operation' => 'update_billing',
      ],
      [
        '_entity_access' => "$entity_type_id.update",
        '_access_check_recurly_user' => 'TRUE',
        '_access_check_recurly_list' => 'TRUE',
        '_access_check_recurly_local_account' => 'TRUE',
      ],
      $options
    );
  }

}
