<?php

namespace Drupal\recurly\Controller;

use Drupal\Core\Utility\Error;
use Recurly\RecurlyError;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for processing Recurly webhooks.
 *
 * See https://recurly.com/developers/reference/webhooks/.
 */
class RecurlyPushListenerController extends RecurlyController {

  /**
   * Process push notification.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   Request stack.
   * @param string $key
   *   Recurly listener key.
   * @param string $subdomain
   *   Recurly account subdomain.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The appropriate Response object.
   */
  public function processPushNotification(Request $request, $key, $subdomain = NULL) {
    // Retrieve the webhook data.
    if ($request->getContentTypeFormat() === 'json') {
      $payload = $request->getContent();
      $notification = json_decode($payload);
    }
    else {
      $notification = NULL;
    }

    // Bail if this is an empty or invalid notification.
    if (empty($notification) || empty($notification->object_type) || empty($notification->event_type)) {
      return new Response('Empty or invalid notification.', Response::HTTP_BAD_REQUEST);
    }

    // Verify the Recurly signature.
    $webhook_endpoint_key = $this->config('recurly.settings')->get('recurly_listener_secret');
    if (!empty($webhook_endpoint_key)) {
      $recurly_signature = $request->headers->get('Recurly-Signature');
      if (!$recurly_signature) {
        // Signature header isn't present.
        $this->getLogger('recurly')->warning('Payload does not contain a Recurly-Signature header.', []);
        return new Response('Missing Recurly-Signature header', Response::HTTP_BAD_REQUEST);
      }

      $computed_signature = hash_hmac('sha256', $payload, $webhook_endpoint_key);
      if (!hash_equals($computed_signature, $recurly_signature)) {
        // Signature is invalid, return an error response.
        $this->getLogger('recurly')->warning('Payload does not pass signature verification.', []);
        return new Response('Invalid signature!', Response::HTTP_BAD_REQUEST);
      }
    }

    // Ensure that the push notification was sent with the proper key.
    if ($key != $this->config('recurly.settings')
      ->get('recurly_listener_key')) {
      // Log the failed attempt and bail.
      $url_key_error_text = 'Incoming push notification did not contain the proper URL key.';
      $this->getLogger('recurly')->warning($url_key_error_text, []);
      return new Response($url_key_error_text, Response::HTTP_FORBIDDEN);
    }

    // Log the incoming push notification if enabled.
    if ($this->config('recurly.settings')->get('recurly_push_logging')) {
      $this->getLogger('recurly')->notice('Incoming %type: <pre>@notification</pre>', [
        '%type' => $notification->object_type . ' -- ' . $notification->event_type,
        '@notification' => print_r($notification, TRUE),
      ]);
    }

    // If this is a new, canceled, or updated account set the database record.
    // account -- created
    // subscription -- created
    // account -- closed
    // subscription -- reactivated
    // billing_info -- updated.
    if (
      ($notification->object_type === 'account' && $notification->event_type === 'created')
      || ($notification->object_type === 'subscription' && $notification->event_type === 'created')
      || ($notification->object_type === 'account' && $notification->event_type === 'closed')
      || ($notification->object_type === 'subscription' && $notification->event_type === 'reactivated')
      || ($notification->object_type === 'billing_info' && $notification->event_type === 'updated')
    ) {

      if (!isset($notification->account_code)) {
        // If this is a notification with no account code, we need to load the
        // object associated with the notification and get the account code from
        // that.
        $account_code = NULL;
        if ($notification->object_type === 'subscription') {
          $subscription = $this->recurlyClient->getSubscription($notification->id);
          $account_code = $subscription->getAccount()->getCode();
        }
      }
      else {
        $account_code = $notification->account_code;
      }

      // Retrieve the full account record from Recurly.
      if ($account_code) {
        try {
          $recurly_account = $this->recurlyClient->getAccount('code-' . $account_code);
        }
        catch (RecurlyError $exception) {
          $logger = $this->loggerFactory->get('recurly');
          Error::logException($logger, $exception);
          return new Response('Recurly account could not be loaded from API', Response::HTTP_BAD_REQUEST);
        }
      }
      else {
        return new Response('Unable to retrieve Recurly account associated with push notification.', Response::HTTP_BAD_REQUEST);
      }

      // Look for a pre-existing local record.
      $local_account = recurly_account_load([
        'account_code' => $recurly_account->getCode(),
      ], TRUE);

      // If no local record exists and we've specified to create it...
      if (empty($local_account)) {
        // First try to match based on the account code.
        // i.e. "user-1" would match the user with UID 1.
        $parts = explode('-', $recurly_account->getCode());
        $entity_type = $this->config('recurly.settings')->get('recurly_entity_type');
        if ($parts[0] === $entity_type) {
          if (isset($parts[1]) && is_numeric($parts[1])) {
            recurly_account_save_local(['code' => $recurly_account->getCode()], $entity_type, $parts[1]);
          }
        }
        // Attempt to find a matching user account by e-mail address if the
        // enabled entity type is user.
        elseif ($entity_type === 'user' && ($user = user_load_by_mail($recurly_account->getEmail()))) {
          recurly_account_save_local(['code' => $recurly_account->getCode()], 'user', $user->id());
        }
      }
      elseif (!empty($local_account)) {
        // Otherwise if a local record was found and we want to keep it
        // synchronized, save it again and let any modules respond.
        recurly_account_save_local(['code' => $recurly_account->getCode()], $local_account->entity_type, $local_account->entity_id);
      }
    }

    // Allow other modules to respond to incoming notifications.
    $this->moduleHandler()->invokeAll('recurly_process_push_notification', [
      $subdomain,
      $notification,
    ]);

    // Reply with the OK status code.
    return new Response('OK', Response::HTTP_OK);
  }

}
