<?php

namespace Drupal\recurly\Controller;

use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Routing\RouteProviderInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\recurly\RecurlyClientFactory;
use Drupal\recurly\RecurlyFormatManager;
use Drupal\recurly\RecurlyPagerManager;
use Recurly\Errors\NotFound;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Recurly select plan controller.
 */
class RecurlySubscriptionSelectPlanController extends RecurlyController {

  const SELECT_PLAN_MODE_SIGNUP = 'signup';

  const SELECT_PLAN_MODE_CHANGE = 'change';

  /**
   * The route provider service.
   *
   * @var \Drupal\Core\Routing\RouteProviderInterface
   */
  protected $routeProvider;

  /**
   * Constructs a new RecurlySubscriptionSelectPlanController object.
   *
   * @param \Drupal\recurly\RecurlyPagerManager $recurly_page_manager
   *   The Recurly page manager service.
   * @param \Drupal\recurly\RecurlyFormatManager $recurly_formatter
   *   The Recurly formatter service.
   * @param \Drupal\recurly\RecurlyClientFactory $clientFactory
   *   The Recurly client factory.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user service.
   * @param \Drupal\Core\Routing\RouteProviderInterface $route_provider
   *   The route provider service.
   */
  public function __construct(RecurlyPagerManager $recurly_page_manager, RecurlyFormatManager $recurly_formatter, RecurlyClientFactory $clientFactory, AccountProxyInterface $current_user, RouteProviderInterface $route_provider) {
    parent::__construct($recurly_page_manager, $recurly_formatter, $clientFactory);
    $this->currentUser = $current_user;
    $this->routeProvider = $route_provider;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('recurly.pager_manager'),
      $container->get('recurly.format_manager'),
      $container->get('recurly.client'),
      $container->get('current_user'),
      $container->get('router.route_provider')
    );
  }

  /**
   * Show a list of available plans to which a user may subscribe.
   *
   * This method is used both for new subscriptions and for updating existing
   * subscriptions.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   A RouteMatch object.
   *   Contains the route and the entity subscription is being changed.
   * @param string $currency
   *   If this is a new subscription, the currency to be used.
   * @param string $subscription_id
   *   The UUID of the current subscription if changing the plan on an existing
   *   subscription.
   *
   * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
   *   A render array for the plan selection page or a redirect response.
   */
  public function planSelect(RouteMatchInterface $route_match, $currency = NULL, $subscription_id = NULL) {
    $entity_type = $this->config('recurly.settings')
      ->get('recurly_entity_type');

    // Redirect authenticated users to the authenticated signup page if they're
    // on the unauthenticated one.
    if ($this->currentUser->isAuthenticated() && !$route_match->getParameters()->count()) {
      $authenticated_route_name = "entity.$entity_type.recurly_signup";
      $authenticated_route = $this->routeProvider->getRouteByName($authenticated_route_name);
      return $this->redirect($authenticated_route_name, [
        'user' => $this->currentUser->id(),
      ], $authenticated_route->getOptions());
    }

    $entity = $route_match->getParameter($entity_type);

    $mode = $subscription_id ? self::SELECT_PLAN_MODE_CHANGE : self::SELECT_PLAN_MODE_SIGNUP;
    $subscriptions = [];

    // If loading an existing subscription.
    if ($subscription_id) {
      if ($subscription_id === 'latest') {
        $local_account = recurly_account_load([
          'entity_type' => $entity_type,
          'entity_id' => $entity->id(),
        ], TRUE);
        $subscriptions = recurly_account_get_subscriptions($local_account->account_code, 'active');
        /** @var \Recurly\Resources\Subscription $subscription */
        $subscription = reset($subscriptions);
        $subscription_id = $subscription->getUuid();
      }
      else {
        try {
          $subscription = $this->recurlyClient->getSubscription($subscription_id);
          $subscriptions[$subscription->getUuid()] = $subscription;
        }
        catch (NotFound $e) {
          throw new NotFoundHttpException('Subscription not found');
        }
      }
      $currency = $subscription->getCurrency();
    }
    // If signing up to a new subscription, ensure the user doesn't have a plan.
    elseif ($this->currentUser->isAuthenticated()) {
      $currency = $currency ?? $this->config('recurly.settings')
        ->get('recurly_default_currency');
      $account = recurly_account_load([
        'entity_type' => $entity_type,
        'entity_id' => $entity->id(),
      ]);
      if ($account) {
        $subscriptions = recurly_account_get_subscriptions($account->getCode(), 'active');
      }
    }

    // Make the list of subscriptions based on plan keys, rather than uuid.
    $plan_subscriptions = [];
    foreach ($subscriptions as $subscription) {
      $plan_subscriptions[$subscription->getPlan()->getCode()] = $subscription;
    }

    $all_plans = recurly_subscription_plans();
    $enabled_plan_keys = $this->config('recurly.settings')
      ->get('recurly_subscription_plans') ?: [];
    $enabled_plans = [];
    foreach ($enabled_plan_keys as $plan_code => $plan_info) {
      /** @var \Recurly\Resources\Plan $plan */
      foreach ($all_plans as $plan) {
        if ($plan_info['status'] == '1' && $plan_code == $plan->getCode()) {
          $enabled_plans[$plan_code] = $plan;
        }
      }
    }

    return [
      '#theme' => [
        'recurly_subscription_plan_select__' . $mode,
        'recurly_subscription_plan_select',
      ],

      '#plans' => $enabled_plans,
      '#entity_type' => $entity_type,
      '#entity' => $entity,
      '#currency' => $currency,
      '#mode' => $mode,
      '#subscriptions' => $plan_subscriptions,
      '#subscription_id' => $subscription_id,
    ];
  }

  /**
   * Redirect anonymous users to registration when attempting to select plans.
   */
  public function redirectToRegistration() {
    $this->messenger()
      ->addWarning($this->t('Create an account, or log in with an existing account, before selecting a plan.'));
    return $this->redirect('user.register');
  }

}
