<?php

namespace Drupal\recurly\Form;

use Drupal\Core\Form\FormStateInterface;
use Recurly\Errors\NotFound;
use Recurly\RecurlyError;

/**
 * Recurly change quantity form.
 */
class RecurlyChangeQuantityForm extends RecurlyNonConfigForm {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'recurly_change_quantity_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $entity_type_id = $this->config('recurly.settings')->get('recurly_entity_type');
    $entity = $this->getRouteMatch()->getParameter($entity_type_id);
    $entity_type = $entity->getEntityType()->id();
    $subscription_id = $this->getRouteMatch()->getParameter('subscription_id');

    $form['#entity_type'] = $entity_type;
    $form['#entity'] = $entity;

    if (!isset($form['#current_subscription'])) {
      try {
        $subscription = $this->recurlyClient->getSubscription('uuid-' . $subscription_id);
      }
      catch (NotFound $e) {
        $this->messenger()->addError('Unable to retrieve subscription information. Please try again or contact an administrator.');
        $this->logger('recurly')->error('Unable to retrieve subscription information: @error', [$e->getMessage()]);
        return $form;
      }

      $form['#current_subscription'] = $subscription;
    }

    if ($form_state->get('do_preview') === TRUE) {
      // Figure out the current rate before we preview the new rate.
      $current_due = $this->recurlyFormatter->formatCurrency(($subscription->getTotal() * $subscription->getQuantity()), $subscription->getCurrency());

      $old_quantity = $subscription->getQuantity();
      $new_quantity = (int) $form_state->get('preview_quantity');

      // Generate a preview of the quantity change and use the resulting
      // preview invoices to calculate the cost of this change.
      $change = ['quantity' => $new_quantity];
      $preview = $this->recurlyClient->previewSubscriptionChange($subscription->getId(), $change);

      $invoices = [];
      $charge_invoice = $preview->getInvoiceCollection()->getChargeInvoice();
      if ($charge_invoice) {
        $invoices[] = [
          '#theme' => 'recurly_invoice',
          '#attached' => [
            'library' => [
              'recurly/recurly.invoice',
            ],
          ],
          '#invoice' => $charge_invoice,
          '#invoice_account' => $charge_invoice->getAccount(),
          '#entity_type' => $entity_type,
          '#entity' => $entity,
          '#error_message' => NULL,
        ];
      }

      $credit_invoices = $preview->getInvoiceCollection()->getCreditInvoices();
      if (!empty($credit_invoices)) {
        foreach ($credit_invoices as $invoice) {
          $invoices[] = [
            '#theme' => 'recurly_invoice',
            '#attached' => [
              'library' => [
                'recurly/recurly.invoice',
              ],
            ],
            '#invoice' => $invoice,
            '#invoice_account' => $invoice->getAccount(),
            '#entity_type' => $entity_type,
            '#entity' => $entity,
            '#error_message' => NULL,
          ];
        }
      }

      $plan_name = $subscription->getPlan()->getName();
      $due_next = $this->recurlyFormatter->formatCurrency($subscription->getTotal() * $new_quantity, $subscription->getCurrency());

      $form['preview'] = [
        '#type' => 'fieldset',
        '#title' => 'Preview changes',
        '#cache' => [
          'max-age' => 0,
        ],
      ];
      $form['preview']['due_note'] = [
        '#type' => 'markup',
        '#markup' => '<p>' . $this->t('You are changing from <strong>@old_quantity x @plan</strong> (@current_due) to <strong>@new_quantity x @plan</strong> (@due_next). Changes take effect immediately.',
            [
              '@old_quantity' => $old_quantity,
              '@plan' => $plan_name,
              '@current_due' => $current_due,
              '@new_quantity' => $new_quantity,
              '@due_next' => $due_next,
            ]) . '</p>',
      ];

      if (count($invoices)) {
        $form['preview']['invoices_note'] = [
          '#type' => 'markup',
          '#markup' => '<p>' . $this->t('In addition to changing the rate above this change will result in the following prorated charges or credits being issued:') . '</p>',
        ];
        $form['preview']['invoices'] = $invoices;
      }
    }

    $form['quantity'] = [
      '#type' => 'textfield',
      '#title' => $this->t('New quantity'),
      '#size' => 3,
      '#default_value' => 1,
    ];

    // Set the default properly, if we have a subscription.
    $form['quantity']['#default_value'] = $subscription->getQuantity();

    $form['actions'] = [
      '#type' => 'actions',
    ];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Preview quantity change'),
      '#name' => 'preview',
    ];

    $form['actions']['confirm'] = [
      '#type' => 'submit',
      '#value' => $this->t('Confirm quantity change'),
      '#name' => 'confirm',
      '#access' => ($form_state->get('do_preview') === TRUE),
    ];

    $form['actions']['cancel'] = $entity->toLink($this->t('Cancel'), 'recurly-subscriptionlist')->toRenderable();

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Make sure we're dealing with a positive integer.
    if ($form_state->getValue('quantity') < 1) {
      $form_state->setErrorByName('quantity', $this->t('Please enter a valid quantity'));
    }

    // The new quantity value must match what was used in the preview when
    // submitting the update.
    if ($form_state->getTriggeringElement()['#name'] === 'confirm') {
      if ($form_state->getValue('quantity') !== $form_state->get('preview_quantity')) {
        $form_state->setErrorByName('quantity', $this->t('Previewed quantity must match submitted quantity. Please update the preview and try again.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    /** @var \Recurly\Resources\Subscription $subscription */
    $subscription = $form['#current_subscription'];

    if ($subscription) {
      if ($form_state->getTriggeringElement()['#name'] === 'preview') {
        // Rebuild the form and display a preview of the change.
        $form_state
          ->set('preview_quantity', $form_state->getValue('quantity'))
          ->set('do_preview', TRUE)
          ->setRebuild();
      }
      elseif ($form_state->getTriggeringElement()['#name'] === 'confirm') {
        $success = FALSE;
        try {
          // Update immediately.
          // @todo allow user to choose when this happens?
          $change = [
            'timeframe' => 'now',
            'quantity' => $form_state->getValue('quantity'),
          ];
          $this->recurlyClient->createSubscriptionChange($subscription->getId(), $change);
          $success = TRUE;
        }
        catch (RecurlyError $e) {
          $this->messenger()->addError('Unable to update subscription quantity. Please try again or contact an administrator.');
          $this->logger('recurly')->error('Unable to update subscription quantity: @error', [$e->getMessage()]);
          $form_state->setRebuild();
        }

        if ($success) {
          $entity_type = $form['#entity_type'];
          $entity = $form['#entity'];
          $this->messenger()
            ->addMessage($this->t('Your subscription has been updated.'));
          $form_state->setRedirect("entity.$entity_type.recurly_subscriptionlist", [$entity_type => $entity->id()]);
        }
      }
    }
  }

}
