<?php

namespace Drupal\recurly;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerTrait;
use Recurly\Client;
use Recurly\RecurlyError;

/**
 * Factory to get instances of the Recurly API \Recurly\Client object.
 */
class RecurlyClientFactory {

  use MessengerTrait;

  const ERROR_MESSAGE_MISSING_API_KEY = 'The Recurly private API key is not configured.';

  const ERROR_MESSAGE_MISSING_SUBDOMAIN = 'The Recurly subdomain is not configured.';

  /**
   * This module's settings.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $moduleSettings;

  /**
   * The logging service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Class Constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_service
   *   The Recurly configuration.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_service
   *   The logger.
   */
  public function __construct(ConfigFactoryInterface $config_service, LoggerChannelFactoryInterface $logger_service) {
    $this->moduleSettings = $config_service->get('recurly.settings');
    $this->logger = $logger_service->get('recurly');
  }

  /**
   * Get an instance of Recurly_Client configured to connect to the API.
   *
   * @param array $account_settings
   *   An array of Recurly account settings including the following keys or NULL
   *   to use the site-wide account settings.
   *   - username: the API username to use
   *   - password: the API password for the given username
   *   - subdomain: the subdomain configured for your Recurly account.
   *   - environment: the current environment of the given account, either
   *     'sandbox' or 'production'.
   *
   * @return bool|\Recurly\Client
   *   A Recurly API client with credentials and settings loaded, or FALSE if
   *   one could not be created.
   */
  public function getClientFromSettings(?array $account_settings = NULL) {
    static $_client = FALSE;

    if (!$_client) {
      // If no settings array was given, use the default account settings.
      if (empty($account_settings)) {
        $account_settings = $this->getDefaultAccountSettings();
      }

      $this->validateSettings($account_settings);

      // If logging is enabled add the logger to the client so that it saves
      // records to the Drupal configured log.
      if ($this->moduleSettings->get('recurly_client_logging_enabled')) {
        $_client = new Client($account_settings['api_key'], $this->logger);
      }
      else {
        $_client = new Client($account_settings['api_key']);
      }
    }

    return $_client;
  }

  /**
   * Ensure that mandatory settings have been entered.
   *
   * @param array $account_settings
   *   Associative array containing settings to validate including 'api_key',
   *   and 'subdomain'.
   *
   * @throws \Recurly\RecurlyError
   */
  protected function validateSettings(array $account_settings) {
    // Ensure that the mandatory settings have been entered.
    if (empty($account_settings['api_key'])) {
      $message = self::ERROR_MESSAGE_MISSING_API_KEY;
      $this->messenger()->addError($message);
      $this->logger->error($message);
      throw new RecurlyError(self::ERROR_MESSAGE_MISSING_API_KEY);
    }
  }

  /**
   * Fetches the default account settings.
   *
   * @return array
   *   Associative array of settings to use when connecting to Recurly API.
   */
  protected function getDefaultAccountSettings() {
    return [
      'api_key' => $this->moduleSettings->get('recurly_private_api_key'),
      'subdomain' => $this->moduleSettings->get('recurly_subdomain'),
      'public_key' => $this->moduleSettings->get('recurly_public_key'),
    ];
  }

}
