<?php

namespace Drupal\recurly;

use Drupal\Component\Utility\Html;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Link;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Recurly\Errors\NotFound;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Service to abstract preprocess hooks.
 */
class RecurlyPreprocess {

  use StringTranslationTrait;

  /**
   * The Recurly API client service.
   *
   * @var \Recurly\Client
   */
  protected $recurlyClient;

  /**
   * The formatting service.
   *
   * @var \Drupal\recurly\RecurlyFormatManager
   */
  protected $recurlyFormatter;

  /**
   * The Recurly settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $recurlySettings;

  /**
   * Request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a \Drupal\recurly\RecurlyPreprocess object.
   *
   * @param \Drupal\recurly\RecurlyClientFactory $recurly_client_factory
   *   Recurly client factory service.
   * @param \Drupal\recurly\RecurlyFormatManager $recurly_formatter
   *   A Recurly formatter object.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   Request stack service.
   */
  public function __construct(RecurlyClientFactory $recurly_client_factory, RecurlyFormatManager $recurly_formatter, ConfigFactoryInterface $config_factory, RequestStack $request_stack) {
    $this->recurlyClient = $recurly_client_factory->getClientFromSettings();
    $this->recurlyFormatter = $recurly_formatter;
    $this->recurlySettings = $config_factory->get('recurly.settings');
    $this->requestStack = $request_stack;
  }

  /**
   * Implements hook_preprocess_recurly_subscription_plan_select().
   */
  public function preprocessRecurlySubscriptionPlanSelect(array &$variables) {
    $plans = $variables['plans'];
    $currency = $variables['currency'];
    $entity_type = $variables['entity_type'];
    $entity = $variables['entity'];
    $subscriptions = $variables['subscriptions'];
    $subscription_id = $variables['subscription_id'];

    $current_subscription = NULL;
    /** @var \Recurly\Resources\Subscription $subscription */
    foreach ($subscriptions as $subscription) {
      if ($subscription->getUuid() === $subscription_id) {
        $current_subscription = $subscription;
        break;
      }
    }

    // If currency is undefined, use the subscription currency.
    if ($current_subscription && empty($currency)) {
      $currency = $current_subscription->getCurrency();
      $variables['currency'] = $currency;
    }

    // Prepare an easy to loop-through list of subscriptions.
    $variables['filtered_plans'] = [];
    /** @var \Recurly\Resources\Plan $plan */
    foreach ($plans as $plan_code => $plan) {
      $setup_fee_amount = NULL;
      $unit_amount = NULL;
      foreach ($plan->getCurrencies() as $setup_currency) {
        if ($setup_currency->getCurrency() === $currency) {
          $setup_fee_amount = $this->recurlyFormatter->formatCurrency($setup_currency->getSetupFee(), $setup_currency->getCurrency(), TRUE);
          $unit_amount = $this->recurlyFormatter->formatCurrency($setup_currency->getUnitAmount(), $setup_currency->getCurrency(), TRUE);
          break;
        }
      }

      $description = $plan->getDescription() ?? '';
      $variables['filtered_plans'][$plan_code] = [
        'plan_code' => Html::escape($plan_code),
        'name' => Html::escape($plan->getName()),
        'description' => Html::escape($description),
        'setup_fee' => $setup_fee_amount,
        'amount' => $unit_amount,
        'plan_interval' => $this->recurlyFormatter->formatPriceInterval($unit_amount, $plan->getIntervalLength(), $plan->getIntervalUnit(), TRUE),
        'trial_interval' => $plan->getTrialLength() ? $this->recurlyFormatter->formatPriceInterval(NULL, $plan->getTrialLength(), $plan->getTrialUnit(), TRUE) : NULL,
        'signup_url' => recurly_url('subscribe', [
          'entity_type' => $entity_type,
          'entity' => $entity,
          'plan_code' => $plan_code,
          'currency' => $currency,
        ]),
        'change_url' => $current_subscription ? recurly_url('change_plan', [
          'entity_type' => $entity_type,
          'entity' => $entity,
          'subscription' => $current_subscription,
          'plan_code' => $plan_code,
        ]) : NULL,
        'selected' => FALSE,
      ];

      // If we have a pending subscription, make that its shown as selected
      // rather than the current active subscription. This should allow users to
      // switch back to a previous plan after making a pending switch to another
      // one.
      foreach ($subscriptions as $subscription) {
        if (!empty($subscription->getPendingChange())) {
          if ($subscription->getPendingChange()->getPlan()->getCode() === $plan_code) {
            $variables['filtered_plans'][$plan_code]['selected'] = TRUE;
          }
        }
        elseif ($subscription->getPlan()->getCode() === $plan_code) {
          $variables['filtered_plans'][$plan_code]['selected'] = TRUE;
        }
      }
    }

    // Check if this is an account that is creating a new subscription.
    $variables['expired_subscriptions'] = FALSE;
    if (!empty($entity) && recurly_account_load([
      'entity_type' => $entity_type,
      'entity_id' => $entity->id(),
    ])) {
      $variables['expired_subscriptions'] = empty($subscriptions);
    }
  }

  /**
   * Implements hook_preprocess_recurly_subscription_cancel_confirm().
   */
  public function preprocessRecurlySubscriptionCancelConfirm(array &$variables) {
    $variables['subscription'] = $variables['form']['#subscription'];
    $variables['past_due'] = $this->requestStack->getCurrentRequest()->query->get('past_due') === '1';
  }

  /**
   * Implements hook_preprocess_recurly_invoice_list().
   */
  public function preprocessRecurlyInvoiceList(array &$variables) {
    $invoices = $variables['invoices'];
    $entity_type = $variables['entity_type'];
    $entity = $variables['entity'];

    $header = [
      $this->t('Number'),
      $this->t('Date'),
      $this->t('Total'),
    ];
    $rows = [];
    /** @var \Recurly\Resources\Invoice $invoice */
    foreach ($invoices as $invoice) {
      $status = ' ';
      if ($invoice->getState() === 'past_due') {
        $status .= $this->t('(Past due)');
      }
      elseif ($invoice->getState() === 'failed') {
        $status .= $this->t('(Failed)');
      }

      $row = [];
      $row[] = Link::createFromRoute($invoice->getNumber() . $status, "entity.$entity_type.recurly_invoice", [
        $entity_type => $entity->id(),
        'invoice_number' => $invoice->getNumber(),
      ]);

      $row[] = $this->recurlyFormatter->formatDate($invoice->getCreatedAt());
      $row[] = $this->recurlyFormatter->formatCurrency($invoice->getTotal(), $invoice->getCurrency());
      $rows[] = [
        'data' => $row,
        'class' => [Html::escape($invoice->getState())],
      ];
    }

    $variables['table'] = [
      '#theme' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#attributes' => ['class' => ['invoice-list']],
      '#sticky' => FALSE,
    ];
    $variables['pager'] = ['#type' => 'pager'];
  }

  /**
   * Implements hook_preprocess_recurly_invoice().
   */
  public function preprocessRecurlyInvoice(array &$variables) {
    $entity_type_id = $this->recurlySettings->get('recurly_entity_type');
    /** @var \Recurly\Resources\Invoice $invoice */
    $invoice = $variables['invoice'];
    $entity = $variables['entity'];

    try {
      $billing_info = $this->recurlyClient->getBillingInfo($invoice->getAccount()->getId());
    }
    catch (NotFound $e) {
      $billing_info = NULL;
    }

    $due_amount = $invoice->getState() !== 'paid' ? $invoice->getTotal() : 0;
    $paid_amount = $invoice->getState() === 'paid' ? $invoice->getTotal() : 0;
    $variables += [
      'invoice_date' => $this->recurlyFormatter->formatDate($invoice->getCreatedAt()),
      'subtotal' => $this->recurlyFormatter->formatCurrency($invoice->getSubtotal(), $invoice->getCurrency()),
      'total' => $this->recurlyFormatter->formatCurrency($invoice->getTotal(), $invoice->getCurrency()),
      'due' => $this->recurlyFormatter->formatCurrency($due_amount, $invoice->getCurrency()),
      'paid' => $this->recurlyFormatter->formatCurrency($paid_amount, $invoice->getCurrency()),
      'billing_info' => isset($billing_info),
      'line_items' => [],
      'transactions' => [],
    ];

    // A 'pending' invoice won't have an invoice number yet.
    if ($invoice->getNumber()) {
      $variables['pdf_link'] = Link::createFromRoute($this->t('View PDF'), "entity.$entity_type_id.recurly_invoicepdf", [
        $entity_type_id => $entity->id(),
        'invoice_number' => $invoice->getNumber(),
      ]);
    }

    if ($billing_info) {
      $variables += [
        'first_name' => Html::escape($billing_info->getFirstName() ?? ''),
        'last_name' => Html::escape($billing_info->getLastName() ?? ''),
        'address1' => Html::escape($billing_info->getAddress()->getStreet1() ?? ''),
        'address2' => Html::escape($billing_info->getAddress()->getStreet2() ?? ''),
        'city' => Html::escape($billing_info->getAddress()->getCity() ?? ''),
        'state' => Html::escape($billing_info->getAddress()->getRegion() ?? ''),
        'zip' => Html::escape($billing_info->getAddress()->getPostalCode() ?? ''),
        'country' => Html::escape($billing_info->getAddress()->getCountry() ?? ''),
      ];
    }

    /** @var \Recurly\Resources\LineItem $line_item */
    foreach ($invoice->getLineItems() as $line_item) {
      $variables['line_items'][$line_item->getUuid()] = [
        'start_date' => $this->recurlyFormatter->formatDate($line_item->getStartDate()),
        'end_date' => $this->recurlyFormatter->formatDate($line_item->getEndDate()),
        'description' => Html::escape($line_item->getDescription() ?? ''),
        'amount' => $this->recurlyFormatter->formatCurrency($line_item->getUnitAmount(), $line_item->getCurrency()),
      ];
    }

    $transaction_total = 0;
    /** @var \Recurly\Resources\Transaction $transaction */
    foreach ($invoice->getTransactions() as $transaction) {
      $variables['transactions'][$transaction->getUuid()] = [
        'date' => $this->recurlyFormatter->formatDate($transaction->getCreatedAt()),
        'description' => $this->recurlyFormatter->formatTransactionStatus($transaction->getStatus()),
        'amount' => $this->recurlyFormatter->formatCurrency($transaction->getAmount(), $transaction->getCurrency()),
      ];
      if ($transaction->getStatus() == 'success') {
        $transaction_total += $transaction->getAmount();
      }
      else {
        $variables['transactions'][$transaction->getUuid()]['amount'] = '(' . $variables['transactions'][$transaction->getUuid()]['amount'] . ')';
      }
    }
    $variables['transactions_total'] = $this->recurlyFormatter->formatCurrency($transaction_total, $invoice->getCurrency());
  }

}
