<?php

namespace Drupal\recurly;

use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Recurly token manager.
 */
class RecurlyTokenManager {

  /**
   * The Recurly settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $recurlySettings;

  /**
   * Constructs the Recurly token manager.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config service.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->recurlySettings = $config_factory->get('recurly.settings');
  }

  /**
   * Get the token mapping for Recurly variables with defaults populated.
   */
  public function tokenMapping() {
    $mapping = $this->recurlySettings->get('recurly_token_mapping') ?: [];
    $mapping += [
      'email' => '[user:mail]',
      'username' => '[user:name]',
      'first_name' => '',
      'last_name' => '',
      'company_name' => '',
      'address1' => '',
      'address2' => '',
      'city' => '',
      'state' => '',
      'zip' => '',
      'country' => '',
      'phone' => '',
    ];
    return $mapping;
  }

  /**
   * Processes different token values for the provided account.
   *
   * @param string $token
   *   The token to be processed.
   * @param string $value
   *   The value associated with the token.
   * @param array $structure
   *   The subscription object to update.
   *
   * @return bool|null
   *   TRUE if the token was processed successfully, NULL if the token is not
   *   recognized.
   */
  public function mapTokenFieldsToRecurlyStructure(string $token, string $value, array &$structure) {
    switch ($token) {
      case 'email':
        $structure['email'] = $value;
        return TRUE;

      case 'username':
        $structure['username'] = $value;
        return TRUE;

      case 'first_name':
        $structure['first_name'] = $value;
        return TRUE;

      case 'last_name':
        $structure['last_name'] = $value;
        return TRUE;

      case 'company_name':
        $structure['company'] = $value;
        return TRUE;

      case 'address1':
        $structure['address'] = $structure['address'] ?? [];
        $structure['address']['street1'] = $value;
        return TRUE;

      case 'address2':
        $structure['address'] = $structure['account']['address'] ?? [];
        $structure['address']['street2'] = $value;
        return TRUE;

      case 'city':
        $structure['address'] = $structure['account']['address'] ?? [];
        $structure['address']['city'] = $value;
        return TRUE;

      case 'state':
        $structure['address'] = $structure['account']['address'] ?? [];
        $structure['address']['region'] = $value;
        return TRUE;

      case 'zip':
        $structure['address'] = $structure['account']['address'] ?? [];
        $structure['address']['postal_code'] = $value;
        return TRUE;

      case 'country':
        $structure['address'] = $structure['account']['address'] ?? [];
        $structure['address']['country'] = $value;
        return TRUE;

      case 'phone':
        $structure['address'] = $structure['account']['address'] ?? [];
        $structure['address']['phone'] = $value;
        return TRUE;

      default:
        return NULL;
    }
  }

}
