<?php

declare(strict_types=1);

namespace Drupal\recurly_test_client;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Recurly\Client;
use Recurly\Errors\NotFound;
use Recurly\RecurlyResource;
use Recurly\Request;
use Recurly\Response;

/**
 * Recurly V3 mock client.
 *
 * Override the Recurly\Client class to return a mock Recurly API client.
 */
class RecurlyV3MockClient extends Client {

  use StringTranslationTrait;

  /**
   * Makes a request to the Recurly API and returns a canned response.
   *
   * @param string $method
   *   The HTTP method to use.
   * @param string $url
   *   The URL to request.
   * @param array $headers
   *   An array of HTTP headers to send with the request. Optional.
   * @param mixed $body
   *   The request body. Optional.
   *
   * @return \Recurly\RecurlyResource
   *   A mock response from the Recurly API.
   *
   * @throws \Recurly\Errors\NotFound
   *   If the client does not know how to make the request.
   */
  public function makeRequest($method, $url, $headers = [], $body = NULL): RecurlyResource {
    $tempStore = self::getTempStore();

    \Drupal::messenger()->addWarning($this->t('RecurlyMockClient - using mock response for @method : @uri with headers @headers and body @body',
      [
        '@method' => strtoupper($method),
        '@uri' => $url,
        '@headers' => print_r($headers, TRUE),
        '@body' => print_r($body, TRUE),
      ]
    ));

    // Log that this request was made.
    $previousRequests = $tempStore->get('request_log');
    $tempStore->delete('request_log');
    $previousRequests[$method][] = $url;
    $tempStore->set('request_log', $previousRequests);

    $storedResponses = $tempStore->get('responses');
    if (isset($storedResponses[$method][$url])) {
      $headers = $storedResponses[$method][$url]['headers'];
      $body = $storedResponses[$method][$url]['body'];
      $response = $this->createMockRecurlyResponse($body, $headers);
      return $response->toResource();
    }
    else {
      \Drupal::messenger()->addWarning($this->t('RecurlyMockClient does not know how to @method : @uri',
        [
          '@method' => strtoupper($method),
          '@uri' => $url,
        ]
      ));

      throw new NotFound('RecurlyMockClient does not know how to ' . $method . ' ' . $url);
    }
  }

  /**
   * Override the base client pagerCount() method and return canned responses.
   *
   * The code in the Recurly\Client bypasses the makeRequest() method when it
   * makes a HEAD request to the API to get a count of results. So, we need to
   * override it to stop it from making that request.
   *
   * This method needs to return a \Recurly\Response object that has the total
   * records count set via the 'Recurly-Total-Records' HTTP header. The easiest
   * way to do that is to just return the same response from the GET request.
   *
   * @param string $path
   *   Request path.
   * @param array|null $options
   *   Additional options.
   *
   * @return \Recurly\Response
   *   The mock response object with the 'Recurly-Total-Records' header set.
   */
  public function pagerCount(string $path, ?array $options = []): Response {
    $resource = $this->makeRequest('GET', $path);
    return $resource->getResponse();
  }

  /**
   * Add a new response fixture.
   *
   * @param string $method
   *   HTTP request method, e.g. 'GET' or 'POST'.
   * @param string $url
   *   API endpoint that the response should be returned for.
   * @param string $response_body
   *   JSON response to return. Should mimic Recurly API responses.
   * @param array $response_headers
   *   Optional array of HTTP response headers.
   *
   * @throws \Drupal\Core\TempStore\TempStoreException
   */
  public static function addResponse(string $method, string $url, string $response_body, array $response_headers = []) {
    $tempStore = self::getTempStore();

    $body_array = json_decode($response_body);
    if (isset($body_array->object) && $body_array->object === 'list') {
      $count = count($body_array->data);
      $response_headers[] = 'Recurly-Total-Records: ' . $count;
      $response_headers[] = 'HTTP/1.1 200 OK';
    }

    $responses = $tempStore->get('responses');
    $responses[$method][$url] = [
      'headers' => $response_headers,
      'body' => $response_body,
    ];
    $tempStore = self::getTempStore();
    $tempStore->delete('responses');
    $tempStore->set('responses', $responses);
  }

  /**
   * Assert that the specified API request has been made.
   *
   * @param string $method
   *   HTTP request method.
   * @param string $uri
   *   URI check if it was requested.
   *
   * @return bool
   *   TRUE if the request is in the call log.
   */
  public static function assertRequestMade($method, $uri) {
    $tempStore = self::getTempStore();
    $previousRequests = $tempStore->get('request_log');
    return (is_array($previousRequests[$method]) && in_array($uri, $previousRequests[$method]));
  }

  /**
   * Clear stored responses and request log.
   */
  public static function clear() {
    $tempStore = self::getTempStore();
    if ($tempStore && $tempStore->get('responses')) {
      $tempStore->delete('responses');
    }
    if ($tempStore && $tempStore->get('request_log')) {
      $tempStore->delete('request_log');
    }
  }

  /**
   * Clear the request log.
   */
  public static function clearRequestLog() {
    $tempStore = self::getTempStore();
    if ($tempStore && $tempStore->get('request_log')) {
      $tempStore->delete('request_log');
    }
  }

  /**
   * Load the temporary storage.
   *
   * @return \Drupal\Core\TempStore\SharedTempStore
   *   Temporary storage object.
   */
  private static function getTempStore() {
    return \Drupal::service('tempstore.shared')
      ->get('recurly_test_client');
  }

  /**
   * Create a mock response for a Recurly API call.
   *
   * Creates a Recurly\Response object with the given data and headers.
   *
   * If no headers are given, the response will be a 200 OK response.
   *
   * @param string $data
   *   The JSON formatted data to include as the content of the response.
   * @param array $headers
   *   Optional headers to include in the response. Defaults to 200 OK.
   *
   * @return \Recurly\Response
   *   The mock response.
   */
  public static function createMockRecurlyResponse($data, array $headers = []) {
    // We need to mock a Request so that  we can use RecurlyResource to generate
    // a resource object. It doesn't matter what's in the request since we're
    // not actually issuing it.
    $options = [
      'params' => [
        'param-1' => 1,
        'param-2' => 'Param 2',
      ],
      'headers' => [
        'header-1' => 'Header 1',
        'header-2' => 'Header 2',
        'core-header-1' => 'Should be overridden',
      ],
      'core_headers' => [
        'core-header-1' => 'Core Header 1',
        'core-header-2' => 'Core Header 2',
      ],
    ];
    $body = [
      'code' => 'account-code',
      'first_name' => 'Wesley',
      'last_name' => 'Elizabeth',
      'email' => 'elbow@example.com',
    ];
    $request = new Request(
      'GET',
      '/accounts',
      $body,
      $options
    );

    $response = new Response($data, $request);

    if (count($headers) === 0) {
      $headers = ['HTTP/1.1 200 OK'];
    }

    $response->setHeaders($headers);
    return $response;
  }

  /**
   * Load JSON fixture data from file.
   *
   * @param string $filename
   *   The path to the file relative to the tests/fixtures/ directory, do not
   *   include the .json extension.
   *
   * @return false|string
   *   The JSON fixture data, or FALSE if the file doesn't exist.
   */
  public static function loadRecurlyJsonFixture(string $filename) {
    $fixture = file_get_contents(__DIR__ . '/../fixtures/' . $filename . '.json');
    return $fixture;
  }

}
