<?php

namespace Drupal\Tests\recurly\Functional;

use Drupal\recurly_test_client\RecurlyV3MockClient;

/**
 * Tests ability to cancel a recurly subscription.
 *
 * @covers \Drupal\recurly\Controller\RecurlySubscriptionCancelController
 * @covers \Drupal\recurly\Form\RecurlySubscriptionCancelConfirmForm
 * @group recurly
 */
class SubscriptionCancelTest extends RecurlyBrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected function setUp() : void {
    parent::setUp();
    $this->drupalPlaceBlock('local_tasks_block');
    $this->drupalPlaceBlock('system_messages_block');
  }

  /**
   * Test that a bad subscription ID returns a 404.
   */
  public function testUserCancelSubscriptionNotFound() {
    RecurlyV3MockClient::clear();
    $error_message = json_encode(['type' => 'not_found', 'message' => 'Subscription not found ']);
    RecurlyV3MockClient::addResponse('GET', '/subscriptions/uuid-non-existent-subscription', $error_message, ['HTTP/1.1 404 Not Found']);

    $account = $this->createUserWithSubscription();
    $this->drupalLogin($account);
    // There's a race condition with this test where sometimes the db
    // transaction started to log the user in isn't completed before the
    // \Drupal\recurly\Controller\RecurlySubscriptionCancelController::subscriptionCancel
    // throws an exception. Which causes the test to fail. So we just visit
    // another page to give it a little longer.
    $this->drupalGet('<front>');
    $this->drupalGet('/user/' . $account->id() . '/subscription/id/non-existent-subscription/cancel');
    $this->assertSession()->statusCodeEquals(404);
    $this->assertSession()->pageTextContains('Subscription not found');
  }

  /**
   * Test user's ability to cancel their subscription at next renewal.
   */
  public function testUserCancelAtRenewal() {
    RecurlyV3MockClient::clear();

    $account = $this->createUserWithSubscription();
    $this->drupalLogin($account);
    $this->addSharedResponses($account);

    $this->config('recurly.settings')
      ->set('recurly_subscription_cancel_behavior', 'cancel')
      ->save();

    // Routes to 'user/' . $account->id() . '/subscription/id/latest/cancel'.
    // Test that the dynamic route which automatically chooses the latest
    // subscription works.
    $this->drupalGet($account->toUrl('recurly-cancellatest'));
    $this->assertSession()->pageTextContains('Cancel subscription');
    $this->assertSession()->buttonExists('Cancel at Renewal');

    // Load the form again, this by do so by navigating there like a user would.
    // This also confirms the UUID version of the URL works.
    $this->drupalGet($account->toUrl());
    $this->clickLink('Subscription');
    $this->assertSession()->pageTextContains('Subscription Summary');
    $this->clickLink('Cancel');
    $this->assertSession()->addressMatches('@^/user/' . $account->id() . '/subscription/id/.*/cancel$@');
    $this->assertSession()->pageTextContains('Cancel subscription');
    $this->assertSession()->buttonExists('Cancel at Renewal');

    // When the form is submitted it'll make a PUT request to Recurly to update
    // the subscription. We need to mock a response for that, and update the
    // subscription GET response to indicate that it's been cancelled in the
    // Recurly backend.
    $this->subscription->state = 'canceled';
    $this->subscription->updated_at = date('c');
    $this->subscription->canceled_at = date('c');
    $this->subscription->expires_at = date('c', strtotime('+2 weeks'));
    RecurlyV3MockClient::addResponse('PUT', '/subscriptions/' . $this->subscription->id . '/cancel', json_encode($this->subscription));

    // /subscriptions/uuid-sub-12345-uuid
    RecurlyV3MockClient::addResponse('GET', '/subscriptions/uuid-' . $this->subscription->uuid, json_encode($this->subscription));

    $list = [
      'object' => 'list',
      'has_more' => FALSE,
      'next' => NULL,
      'data' => [$this->subscription],
    ];
    RecurlyV3MockClient::addResponse('GET', '/accounts/user-' . $account->id() . '/subscriptions', json_encode($list));

    $this->getSession()->getPage()->pressButton('Cancel at Renewal');
    $this->assertSession()->pageTextContains('Plan Silver Plan canceled!');
    $this->assertSession()->elementTextContains('css', '.subscription .status', 'Canceled (will not renew)');
    $this->assertTrue(RecurlyV3MockClient::assertRequestMade('PUT', '/subscriptions/' . $this->subscription->id . '/cancel'));
  }

  /**
   * Test ability to terminate subscription immediately with partial refund.
   */
  public function testUserCancelTerminateProratedRefund() {
    RecurlyV3MockClient::clear();

    $account = $this->createUserWithSubscription();
    $this->drupalLogin($account);
    $this->addSharedResponses($account);

    $this->config('recurly.settings')
      ->set('recurly_subscription_cancel_behavior', 'terminate_prorated')
      ->save();

    $recurly_account = recurly_account_load([
      'entity_type' => $account->bundle(),
      'entity_id' => $account->id(),
    ], TRUE);
    $subscriptions = recurly_account_get_subscriptions($recurly_account->account_code, 'active');
    /** @var \Recurly\Resources\Subscription $subscription */
    $subscription = reset($subscriptions);

    // Load the page directly. We tested the navigation elsewhere.
    $this->drupalGet('/user/' . $account->id() . '/subscription/id/' . $subscription->getUuid() . '/cancel');
    $this->assertSession()->pageTextContains('Cancel subscription');

    // $unit_amount_in_cents * $remaining_time / $total_period_time
    // See recurly_subscription_calculate_refund()
    $this->assertSession()->pageTextMatches('/A refund of \\$50\..* USD will be credited to your account/');
    $this->assertSession()->buttonExists('Cancel Plan');

    // When the cancel form is submitted it'll trigger a DELETE call to Recurly
    // for a terminiation request. Which we need to mock a response for. As well
    // as change the response that is returned when retrieving the subscription
    // so that it now shows that it's been terminated. This mocks logic that
    // happens in Recurly as a result of the DELETE request.
    $this->subscription->state = 'expired';
    $this->subscription->updated_at = date('c');
    $this->subscription->canceled_at = date('c');
    $this->subscription->expires_at = date('c', strtotime('+2 weeks'));
    RecurlyV3MockClient::addResponse('DELETE', '/subscriptions/' . $this->subscription->id, json_encode($this->subscription));

    $list = [
      'object' => 'list',
      'has_more' => FALSE,
      'next' => NULL,
      'data' => [$this->subscription],
    ];
    RecurlyV3MockClient::addResponse('GET', '/accounts/user-' . $account->id() . '/subscriptions', json_encode($list));

    // Submit the form. And verify the updated summary page.
    $this->getSession()->getPage()->pressButton('Cancel Plan');
    $this->assertSession()->pageTextContains('Plan Silver Plan terminated!');
    $this->assertSession()->elementTextContains('css', '.subscription .status', 'Expired');
    // Also verify that a PUT request gets made to Recurly.
    $this->assertTrue(RecurlyV3MockClient::assertRequestMade('DELETE', '/subscriptions/' . $this->subscription->id));
  }

  /**
   * Test ability to terminate subscription immediately with full refund.
   */
  public function testUserCancelTerminateFullRefund() {
    RecurlyV3MockClient::clear();

    $account = $this->createUserWithSubscription();
    $this->drupalLogin($account);
    $this->addSharedResponses($account);

    $this->config('recurly.settings')
      ->set('recurly_subscription_cancel_behavior', 'terminate_full')
      ->save();

    $recurly_account = recurly_account_load([
      'entity_type' => $account->bundle(),
      'entity_id' => $account->id(),
    ], TRUE);
    $subscriptions = recurly_account_get_subscriptions($recurly_account->account_code, 'active');
    /** @var \Recurly\Resources\Subscription $subscription */
    $subscription = reset($subscriptions);

    // Load the page directly. We tested the navigation elsewhere.
    $this->drupalGet('/user/' . $account->id() . '/subscription/id/' . $subscription->getUuid() . '/cancel');
    $this->assertSession()->pageTextContains('Cancel subscription');

    // See recurly_subscription_calculate_refund()
    $this->assertSession()->pageTextContains('A refund of $100.00 USD will be credited to your account.');
    $this->assertSession()->buttonExists('Cancel Plan');

    // When the cancel form is submitted it'll trigger a DELETE call to Recurly
    // which we need to mock a response for. As well as change the response
    // that is returned when retrieving the subscription so that it now shows
    // that it's been terminated. This mocks logic that happens in Recurly as
    // a result of the DELETE request.
    $this->subscription->state = 'expired';
    $this->subscription->updated_at = date('c');
    $this->subscription->canceled_at = date('c');
    $this->subscription->expires_at = date('c');
    RecurlyV3MockClient::addResponse('DELETE', '/subscriptions/' . $this->subscription->id, json_encode($this->subscription));

    $list = [
      'object' => 'list',
      'has_more' => FALSE,
      'next' => NULL,
      'data' => [$this->subscription],
    ];
    RecurlyV3MockClient::addResponse('GET', '/accounts/user-' . $account->id() . '/subscriptions', json_encode($list));

    // Submit the form. And verify the updated summary page.
    $this->getSession()->getPage()->pressButton('Cancel Plan');
    $this->assertSession()->pageTextContains('Plan Silver Plan terminated!');
    $this->assertSession()->elementTextContains('css', '.subscription .status', 'Expired');
    // Also verify that a PUT request gets made to Recurly.
    $this->assertTrue(RecurlyV3MockClient::assertRequestMade('DELETE', '/subscriptions/' . $this->subscription->id));
  }

  /**
   * Test ability to terminate subscription immediately with full refund.
   */
  public function testAdminCancelOptions() {
    RecurlyV3MockClient::clear();

    $account = $this->createUserWithSubscription();
    $this->addSharedResponses($account);

    $recurly_account = recurly_account_load([
      'entity_type' => $account->bundle(),
      'entity_id' => $account->id(),
    ], TRUE);
    $subscriptions = recurly_account_get_subscriptions($recurly_account->account_code, 'active');
    /** @var \Recurly\Resources\Subscription $subscription */
    $subscription = reset($subscriptions);

    $admin = $this->drupalCreateUser(['administer recurly', 'administer users']);
    $this->drupalLogin($admin);

    // Admin should see all cancellation options regardless of configuration.
    $this->drupalGet('/user/' . $account->id() . '/subscription/id/' . $subscription->getUuid() . '/cancel');
    $this->assertSession()->pageTextContains('Cancel subscription');
    $this->assertSession()->buttonExists('Cancel at Renewal');
    $this->assertSession()->pageTextContains('USD - None');
    $this->assertSession()->pageTextContains('USD - Prorated');
    $this->assertSession()->pageTextContains('USD - Full');
    $this->assertSession()->buttonExists('Terminate Immediately');

    // When the form is submitted it'll make a PUT request to Recurly to update
    // the subscription. We need to mock a response for that, and update the
    // subscription GET response to indicate that it's been cancelled in the
    // Recurly backend.
    $this->subscription->state = 'canceled';
    $this->subscription->updated_at = date('c');
    $this->subscription->canceled_at = date('c');
    $this->subscription->expires_at = date('c', strtotime('+2 weeks'));
    RecurlyV3MockClient::addResponse('PUT', '/subscriptions/' . $this->subscription->id . '/cancel', json_encode($this->subscription));

    $list = [
      'object' => 'list',
      'has_more' => FALSE,
      'next' => NULL,
      'data' => [$this->subscription],
    ];
    RecurlyV3MockClient::addResponse('GET', '/accounts/user-' . $account->id() . '/subscriptions', json_encode($list));

    // Clicking the 'Cancel at Renewal' button should work even if none of the
    // required radio options in the form are selected. This tests the
    // #limit_validation_options feature on that button.
    $this->getSession()->getPage()->pressButton('Cancel at Renewal');
    $this->assertSession()->pageTextContains('Plan Silver Plan canceled!');
  }

}
