<?php

namespace Drupal\Tests\recurly\Functional;

use Drupal\recurly_test_client\RecurlyV3MockClient;

/**
 * Tests ability to change a recurly subscription plan.
 *
 * @group recurly
 */
class SubscriptionChangeTest extends RecurlyBrowserTestBase {

  /**
   * User with subscription to silver plan.
   *
   * @var \Drupal\user\Entity\User
   */
  protected $user;

  /**
   * Current subscription object.
   *
   * @var \Recurly\Resources\Subscription
   */
  protected $subscription;

  /**
   * {@inheritdoc}
   */
  protected function setUp() : void {
    parent::setUp();
    $this->drupalPlaceBlock('local_tasks_block');
    $this->drupalPlaceBlock('system_messages_block');

    // Enable both silver and bedrock mock plans.
    $this->config('recurly.settings')
      ->set('recurly_subscription_plans', [
        'silver' => [
          'status' => 1,
          'weight' => 0,
        ],
        'bedrock' => [
          'status' => 1,
          'weight' => 1,
        ],
      ])
      ->save();

    RecurlyV3MockClient::clear();

    // Create a user with a subscription to the silver plan.
    $this->user = $this->createUserWithSubscription();
    $this->addSharedResponses($this->user);
    $this->drupalLogin($this->user);
  }

  /**
   * Test proper error handling for routes.
   *
   * @covers \Drupal\recurly\Controller\RecurlySubscriptionChangeController::changePlan
   */
  public function testRoutingErrorHandling() {
    // Test loading a bad UUID.
    $this->drupalGet('/user/' . $this->user->id() . '/subscription/id/BAD-UUID/change/bedrock');
    $this->assertSession()->statusCodeEquals(404);
    $this->assertSession()->pageTextContains('Subscription not found');

    // Test what happens if trying to load the current/old plan doesn't work.
    // The old plan code is the plan associated with the current subscription,
    // which for our mock subscription is the "silver" plan.
    $error_message = json_encode(['type' => 'not_found', 'message' => 'Plan not found']);
    RecurlyV3MockClient::addResponse('GET', '/plans/silver', $error_message, ['HTTP/1.1 404 Not Found']);
    $this->drupalGet('/user/' . $this->user->id() . '/subscription/id/' . $this->subscription->uuid . '/change/bedrock');
    $this->assertSession()->statusCodeEquals(404);
    $this->assertSession()->pageTextContains('Plan code "silver" not found');

    // Test loading bad new plan code.
    // The new plan code is specified in the URL when changing plans.
    $error_message = json_encode(['type' => 'not_found', 'message' => 'Plan not found']);
    RecurlyV3MockClient::addResponse('GET', '/plans/bad-plan-code', $error_message, ['HTTP/1.1 404 Not Found']);
    $plan = RecurlyV3MockClient::loadRecurlyJsonFixture('plans/show-200');
    RecurlyV3MockClient::addResponse('GET', '/plans/silver', $plan);
    $this->drupalGet('/user/' . $this->user->id() . '/subscription/id/' . $this->subscription->uuid . '/change/bad-plan-code');
    $this->assertSession()->statusCodeEquals(404);
    $this->assertSession()->pageTextContains('Plan code "bad-plan-code" not found');
  }

  /**
   * Test ability for user to see plans they can change too.
   */
  public function testChangePlanList() {
    $silver_plan = json_decode(RecurlyV3MockClient::loadRecurlyJsonFixture('plans/show-200'));
    $bedrock_plan = clone $silver_plan;
    $bedrock_plan->id = 'bedrock';
    $bedrock_plan->code = 'bedrock';
    $bedrock_plan->description = 'Bedrock is at the bottom.';
    $plan_list = [
      'object' => 'list',
      'has_more' => FALSE,
      'next_page' => NULL,
      'data' => [$silver_plan, $bedrock_plan],
    ];
    RecurlyV3MockClient::addResponse('GET', '/plans', json_encode($plan_list));

    $this->drupalGet('/user/' . $this->user->id() . '/subscription/change');
    $this->assertSession()->pageTextContains('Change plan');
    $this->assertSession()
      ->elementTextContains('css', '.plan-silver', 'Selected');
    $this->assertSession()
      ->elementTextContains('css', '.plan-bedrock', 'Select');
  }

  /**
   * Upgrade to a more expensive plan immediately.
   *
   * This one also confirms the logic for a plan change. The other tests just
   * verify wording on the confirmation form since the logic after submission
   * is the same regardless.
   *
   * @covers \Drupal\recurly\Form\RecurlySubscriptionChangeConfirmForm
   * @covers \Drupal\recurly\Form\RecurlySubscriptionChangeConfirmForm::submitForm
   * @covers \Drupal\recurly\Controller\RecurlySubscriptionChangeController::changePlan
   */
  public function testUpgradeImmediate() {
    $silver_plan = json_decode(RecurlyV3MockClient::loadRecurlyJsonFixture('plans/show-200'));
    RecurlyV3MockClient::addResponse('GET', '/plans/silver', json_encode($silver_plan));

    $bedrock_plan = clone $silver_plan;
    $bedrock_plan->id = 'bedrock';
    $bedrock_plan->code = 'bedrock';
    $bedrock_plan->name = 'Bedrock Plan';
    $bedrock_plan->description = 'Bedrock is at the bottom.';
    RecurlyV3MockClient::addResponse('GET', '/plans/code-bedrock', json_encode($bedrock_plan));

    $plan_list = [
      'object' => 'list',
      'has_more' => FALSE,
      'next_page' => NULL,
      'data' => [$silver_plan, $bedrock_plan],
    ];
    RecurlyV3MockClient::addResponse('GET', '/plans', json_encode($plan_list));

    $this->config('recurly.settings')
      ->set('recurly_subscription_upgrade_timeframe', 'now')
      ->save();

    $this->drupalGet('/user/' . $this->user->id() . '/subscription/id/' . $this->subscription->uuid . '/change/bedrock');
    $this->assertSession()->pageTextContains('The new plan will take effect immediately and a prorated charge (or credit) will be applied to this account.');

    // Verify an error is displayed if we get no response from Recurly.
    $this->getSession()->getPage()->pressButton('Change plan');
    $this->assertTrue(RecurlyV3MockClient::assertRequestMade('POST', '/subscriptions/' . $this->subscription->id . '/change'));
    $this->assertSession()->pageTextContains('The plan could not be updated because the billing service encountered an error.');

    // Now mock the response and try again.
    RecurlyV3MockClient::clearRequestLog();
    $change = clone $this->subscription;
    $change->object = 'subscription_change';
    $change->plan = $bedrock_plan;
    RecurlyV3MockClient::addResponse('POST', '/subscriptions/' . $this->subscription->id . '/change', json_encode($change));
    $this->getSession()->getPage()->pressButton('Change plan');
    $this->assertTrue(RecurlyV3MockClient::assertRequestMade('POST', '/subscriptions/' . $this->subscription->id . '/change'));
    $this->assertSession()->pageTextContains('Plan changed to Bedrock Plan!');
  }

  /**
   * Upgrade to a more expensive plan on renewal.
   *
   * @covers \Drupal\recurly\Form\RecurlySubscriptionChangeConfirmForm
   */
  public function testUpgradeOnRenewal() {
    $silver_plan = json_decode(RecurlyV3MockClient::loadRecurlyJsonFixture('plans/show-200'));
    RecurlyV3MockClient::addResponse('GET', '/plans/silver', json_encode($silver_plan));

    $bedrock_plan = clone $silver_plan;
    $bedrock_plan->id = 'bedrock';
    $bedrock_plan->code = 'bedrock';
    $bedrock_plan->name = 'Bedrock Plan';
    $bedrock_plan->description = 'Bedrock is at the bottom.';
    RecurlyV3MockClient::addResponse('GET', '/plans/code-bedrock', json_encode($bedrock_plan));

    $this->config('recurly.settings')
      ->set('recurly_subscription_upgrade_timeframe', 'renewal')
      ->save();

    $this->drupalGet('/user/' . $this->user->id() . '/subscription/id/' . $this->subscription->uuid . '/change/bedrock');
    $this->assertSession()->pageTextContains('The new plan will take effect on the next billing cycle.');
  }

  /**
   * Verify messaging for downgrades is accurate.
   *
   * @covers \Drupal\recurly\Form\RecurlySubscriptionChangeConfirmForm
   */
  public function testDowngradeMessaging() {
    $silver_plan = json_decode(RecurlyV3MockClient::loadRecurlyJsonFixture('plans/show-200'));
    RecurlyV3MockClient::addResponse('GET', '/plans/silver', json_encode($silver_plan));

    $bedrock_plan = clone $silver_plan;
    $bedrock_plan->id = 'bedrock';
    $bedrock_plan->code = 'bedrock';
    $bedrock_plan->name = 'Bedrock Plan';
    $bedrock_plan->description = 'Bedrock is at the bottom.';
    // Make the bedrock plan cheaper so that it is a downgrade and not an
    // upgrade.
    $bedrock_plan->currencies[0]->unit_amount = 5.00;
    RecurlyV3MockClient::addResponse('GET', '/plans/code-bedrock', json_encode($bedrock_plan));

    $this->config('recurly.settings')
      ->set('recurly_subscription_downgrade_timeframe', 'now')
      ->save();

    $this->drupalGet('/user/' . $this->user->id() . '/subscription/id/' . $this->subscription->uuid . '/change/bedrock');
    $this->assertSession()->pageTextContains('The new plan will take effect immediately and a prorated charge (or credit) will be applied to this account.');

    $this->config('recurly.settings')
      ->set('recurly_subscription_downgrade_timeframe', 'renewal')
      ->save();

    $this->drupalGet('/user/' . $this->user->id() . '/subscription/id/' . $this->subscription->uuid . '/change/bedrock');
    $this->assertSession()->pageTextContains('The new plan will take effect on the next billing cycle.');
  }

  /**
   * Test access to quantity change form based on module configuration.
   *
   * @covers \Drupal\recurly\Access\RecurlyAccessQuantity
   */
  public function testChangeQuantityAccess() {
    $this->drupalLogin($this->user);

    $this->config('recurly.settings')
      ->set('recurly_subscription_multiple', 0)
      ->save();

    // We need a Recurly\Response\Subscription object to pass to the recurly_url
    // function.
    $subscriptionResponse = RecurlyV3MockClient::createMockRecurlyResponse(json_encode($this->subscription));

    $url = recurly_url('quantity', [
      'entity_type' => $this->user->bundle(),
      'entity' => $this->user,
      'subscription' => $subscriptionResponse->toResource(),
    ]);
    $this->drupalGet($url);
    $this->assertSession()->statusCodeEquals(403);

    $this->config('recurly.settings')
      ->set('recurly_subscription_multiple', 1)
      ->save();

    $this->drupalGet($url);
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->fieldExists('quantity');
  }

  /**
   * Test that change quantity form shows an error if subscription is not found.
   *
   * @covers \Drupal\recurly\Form\RecurlyChangeQuantityForm
   */
  public function testChangeQuantityFormSubscriptionNotFound() {
    // We need a Recurly\Response\Subscription object to pass to the recurly_url
    // function.
    $subscriptionResponse = RecurlyV3MockClient::createMockRecurlyResponse(json_encode($this->subscription));

    $this->config('recurly.settings')
      ->set('recurly_subscription_multiple', 1)
      ->save();

    $url = recurly_url('quantity', [
      'entity_type' => $this->user->bundle(),
      'entity' => $this->user,
      'subscription' => $subscriptionResponse->toResource(),
    ]);

    // Error is displayed if attempt to load current subscription fails.
    $error_message = json_encode(['type' => 'not_found', 'message' => 'Subscription not found ']);
    RecurlyV3MockClient::addResponse('GET', '/subscriptions/uuid-' . $this->subscription->uuid, $error_message, ['HTTP/1.1 404 Not Found']);
    $this->drupalGet($url);
    $this->assertTrue(RecurlyV3MockClient::assertRequestMade('GET', '/subscriptions/uuid-' . $this->subscription->uuid));
    $this->assertSession()->pageTextContains('Unable to retrieve subscription information.');
  }

  /**
   * Test the change quantity form.
   *
   * @covers \Drupal\recurly\Form\RecurlyChangeQuantityForm
   */
  public function testChangeQuantityForm() {
    // We need a Recurly\Response\Subscription object to pass to the recurly_url
    // function.
    $subscriptionResponse = RecurlyV3MockClient::createMockRecurlyResponse(json_encode($this->subscription));

    $this->config('recurly.settings')
      ->set('recurly_subscription_multiple', 1)
      ->save();

    $url = recurly_url('quantity', [
      'entity_type' => $this->user->bundle(),
      'entity' => $this->user,
      'subscription' => $subscriptionResponse->toResource(),
    ]);

    $this->drupalGet($url);
    $this->assertSession()->fieldValueEquals('quantity', 1);

    // This should fail because you can't have negative quantity.
    $this->submitForm(['quantity' => -5], 'Preview quantity change');
    $this->assertSession()->pageTextContains('Please enter a valid quantity');

    // Submit again with a valid quantity and it should create a confirmation
    // step with a preview.
    $change = clone $this->subscription;
    $change->object = 'subscription_change';
    $change->quantity = 5;

    $invoice = json_decode(RecurlyV3MockClient::loadRecurlyJsonFixture('invoices/show-200'));

    // Convert object to stdClass;.
    $invoice_collection = new \stdClass();
    $invoice_collection->object = 'invoice_collection';
    $invoice->state = 'pending';
    $credit_invoice = clone $invoice;
    $credit_invoice->total = 0;
    $invoice_collection->charge_invoice = $invoice;
    $invoice_collection->credit_invoice = $credit_invoice;

    $change->invoice_collection = $invoice_collection;
    RecurlyV3MockClient::addResponse('POST', '/subscriptions/' . $this->subscription->id . '/change/preview', json_encode($change));
    $this->submitForm(['quantity' => 5], 'Preview quantity change');
    $this->assertSession()->pageTextContains('Preview changes');
    // Pricing data from fixtures. The actual numbers don't really matter, it
    // just shows that the code will display the data returned from Recurly.
    $this->assertSession()->pageTextContains('You are changing from 1 x Silver Plan ($115.50 USD) to 5 x Silver Plan ($577.50 USD).');
    // Verifies the preview invoice from the fixture is displayed.
    $this->assertSession()->elementTextContains('css', '.invoice', 'Total Due: $155.00 USD');

    // If you try and submit the confirmation form with a quantity that's
    // different than the preview it should re-build the confirmation instead.
    $this->submitForm(['quantity' => 6], 'Confirm quantity change');
    $this->assertSession()->pageTextContains('Previewed quantity must match submitted quantity. Please update the preview and try again.');

    $this->submitForm(['quantity' => 6], 'Preview quantity change');
    // The quantity in this string is generated from the form value not the
    // Recurly API fixture so this check should work.
    $this->assertSession()->pageTextContains('You are changing from 1 x Silver Plan ($115.50 USD) to 6 x Silver Plan ($693.00 USD).');

    // This should result in an error.
    $error_message = json_encode(['type' => 'not_found', 'message' => 'Subscription not found ']);
    RecurlyV3MockClient::addResponse('POST', '/subscriptions/' . $this->subscription->id . '/change', $error_message, ['HTTP/1.1 404 Not Found']);
    $this->submitForm(['quantity' => 6], 'Confirm quantity change');
    $this->assertSession()->pageTextContains('Unable to update subscription quantity.');

    // But this should work.
    $change = [
      'object' => 'subscription_change',
      'id' => $this->subscription->id,
      'quantity' => 6,
    ];
    RecurlyV3MockClient::addResponse('POST', '/subscriptions/' . $this->subscription->id . '/change', json_encode($change));

    // Note that the resulting page will still display the old invoice on it
    // because we didn't update the GET subscription data. But we also don't
    // need to for this test.
    $this->submitForm(['quantity' => 6], 'Confirm quantity change');
    $this->assertSession()->pageTextContains('Your subscription has been updated.');
  }

}
