<?php

namespace Drupal\Tests\recurly\Functional;

use Drupal\Core\Url;
use Drupal\recurly_test_client\RecurlyV3MockClient;

/**
 * Tests ability to reactivate a canceled recurly subscription.
 *
 * @covers \Drupal\recurly\Controller\RecurlySubscriptionReactivateController
 * @group recurly
 */
class SubscriptionReactivateTest extends RecurlyBrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected function setUp() : void {
    parent::setUp();
    $this->drupalPlaceBlock('local_tasks_block');
    $this->drupalPlaceBlock('system_messages_block');
  }

  /**
   * Test that a cancelled subscription can be reactivated.
   */
  public function testReactivateCancelledSubscription() {
    // Create a user with a subscription.
    $account = $this->createUserWithSubscription();
    $this->drupalLogin($account);
    $this->addSharedResponses($account);

    // Add a subscription that can be returned for both getSubscription() and
    // listAccountSubscriptions() for the account.
    $recurly_account_code = 'user-' . $account->id();
    $recurly_subscription = json_decode(RecurlyV3MockClient::loadRecurlyJsonFixture('subscriptions/show-200'));
    $recurly_subscription->account_code = $recurly_account_code;
    $recurly_subscription->state = 'canceled';
    $recurly_subscription->current_period_started_at = date('c', strtotime('-2 weeks'));
    $recurly_subscription->current_period_ends_at = date('c', strtotime('+2 weeks'));
    RecurlyV3MockClient::addResponse('GET', '/subscriptions/' . $recurly_subscription->id, json_encode($recurly_subscription));
    RecurlyV3MockClient::addResponse('GET', '/subscriptions/uuid-' . $recurly_subscription->uuid, json_encode($recurly_subscription));
    $subscription_list = [
      'object' => 'list',
      'has_more' => FALSE,
      'next' => NULL,
      'data' => [$recurly_subscription],
    ];
    RecurlyV3MockClient::addResponse('GET', '/accounts/' . $recurly_account_code . '/subscriptions', json_encode($subscription_list));
    RecurlyV3MockClient::addResponse('GET', '/accounts/code-' . $recurly_account_code . '/subscriptions', json_encode($subscription_list));

    // And, what happens if we return a valid response.
    $recurly_subscription->state = 'active';
    RecurlyV3MockClient::addResponse('PUT', '/subscriptions/' . $recurly_subscription->id . '/reactivate', json_encode($recurly_subscription));

    // Try using the 'latest' subscription route.
    $this->drupalGet($account->toUrl('recurly-reactivatelatest'));
    $this->assertTrue(RecurlyV3MockClient::assertRequestMade('PUT', '/subscriptions/' . $recurly_subscription->id . '/reactivate'));
    $this->assertSession()->pageTextContains('Plan Silver Plan reactivated! Normal billing will resume');

    // Try using the specific subscription ID route.
    $url = Url::fromRoute('entity.user.recurly_reactivate', [
      'user' => $account->id(),
      'subscription_id' => $recurly_subscription->uuid,
    ], ['absolute' => TRUE]);

    // Happy path.
    $this->drupalGet($url->toString());
    $this->assertSession()->pageTextContains('Plan Silver Plan reactivated! Normal billing will resume');

    // Test what happens if we return an invalid response from Recurly for the
    // PUT operation.
    $error_message = json_encode(['type' => 'not_found', 'message' => 'Plan not found']);
    RecurlyV3MockClient::addResponse('PUT', '/subscriptions/' . $recurly_subscription->id . '/reactivate', $error_message, ['HTTP/1.1 404 Not Found']);

    // "Latest" route.
    $this->drupalGet($account->toUrl('recurly-reactivatelatest'));
    $this->assertSession()->pageTextContains('The plan could not be reactivated');

    // Specific subscription route when Recurly returns an invalid response
    // when trying to reactivate the subscription.
    $this->drupalGet($url->toString());
    $this->assertSession()->pageTextContains('The plan could not be reactivated');

    // Specific subscription route with invalid subscription ID should return a
    // 404.
    RecurlyV3MockClient::addResponse('GET', '/subscriptions/' . $recurly_subscription->id, $error_message, ['HTTP/1.1 404 Not Found']);
    RecurlyV3MockClient::addResponse('GET', '/subscriptions/uuid-' . $recurly_subscription->uuid, $error_message, ['HTTP/1.1 404 Not Found']);
    $this->drupalGet($url->toString());
    $this->assertSession()->statusCodeEquals(404);
    $this->assertSession()->pageTextContains('Subscription not found');
  }

}
