<?php

namespace Drupal\Tests\recurly\Functional;

use Drupal\recurly_test_client\RecurlyV3MockClient;
use Drupal\user\UserInterface;

/**
 * Tests ability for users to register an account with a recurly subscription.
 *
 * @group recurly
 */
class UserRegistrationTest extends RecurlyBrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected function setUp() : void {
    parent::setUp();

    $config = $this->config('user.settings');
    // Don't require email verification and allow registration by site visitors
    // without administrator approval.
    $config
      ->set('verify_mail', FALSE)
      ->set('register', UserInterface::REGISTER_VISITORS)
      ->save();

    $silver_plan = json_decode(RecurlyV3MockClient::loadRecurlyJsonFixture('plans/show-200'));
    $plan_list = [
      'object' => 'list',
      'has_more' => FALSE,
      'next' => NULL,
      'data' => [
        $silver_plan,
      ],
    ];
    RecurlyV3MockClient::addResponse('GET', '/plans', json_encode($plan_list));
  }

  /**
   * Test user registration when the recurly entity type is 'user'.
   */
  public function testUserRegistrationRecurlyEntityUser() {
    // Associate Recurly subscriptions with user entities (this is the default).
    // And enable the "silver" mock plan.
    $this->config('recurly.settings')
      ->set('recurly_entity_type', 'user')
      ->set('recurly_subscription_plans', [
        'silver' => [
          'status' => 1,
          'weight' => 0,
        ],
      ])
      ->save();

    $this->rebuildRecurlyEntityRouting();

    $edit = [];
    $edit['name'] = $name = $this->randomMachineName();
    $edit['mail'] = $mail = $edit['name'] . '@example.com';
    $edit['pass[pass1]'] = $new_pass = $this->randomMachineName();
    $edit['pass[pass2]'] = $new_pass;
    $this->drupalGet('user/register');
    $this->submitForm($edit, 'Create new account');
    $this->container->get('entity_type.manager')
      ->getStorage('user')
      ->resetCache();
    $accounts = $this->container->get('entity_type.manager')->getStorage('user')
      ->loadByProperties(['name' => $name, 'mail' => $mail]);

    /** @var \Drupal\user\UserInterface $new_user */
    $new_user = reset($accounts);
    $this->assertNotNull($new_user, 'New account successfully created.');
    $this->assertSession()
      ->pageTextContains('Registration successful. You are now logged in.');

    // Verify the user is redirected to the recurly signup page after
    // registration.
    // @see recurly_user_edit_form_submit_redirect()
    $url = $new_user->toUrl('recurly-signup');
    $this->assertSession()->addressEquals($url);
    $this->assertSession()->pageTextContains('Silver Plan');

    $this->drupalLogout();
  }

}
