<?php

namespace Drupal\Tests\recurly\Kernel;

use Drupal\Core\Messenger\MessengerInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\recurly\Traits\RecurlyTestTrait;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Prophecy\Argument;
use Prophecy\PhpUnit\ProphecyTrait;
use Recurly\Client;
use Recurly\Errors\NotFound;
use Recurly\RecurlyError;

/**
 * Tests for RecurlyEntityOperations service.
 *
 * @covers \Drupal\recurly\RecurlyEntityOperations
 * @group recurly
 */
class RecurlyEntityOperationsTest extends KernelTestBase {

  use ProphecyTrait;
  use RecurlyTestTrait;
  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'recurly',
    'system',
    'user',
  ];

  /**
   * Drupal user account for testing.
   *
   * @var \Drupal\user\Entity\User
   */
  protected $user;

  /**
   * {@inheritdoc}
   */
  protected function setUp() : void {
    parent::setUp();

    $this->installConfig(['recurly', 'user', 'system']);
    $this->config('recurly.settings')
      ->set('recurly_entity_type', 'user')
      ->set('recurly_subscription_plans',
        ['silver' => ['status' => '1', 'weight' => '0']])
      ->save();

    $this->rebuildRecurlyEntityRouting();

    $this->installSchema('recurly', ['recurly_account']);
    $this->installEntitySchema('user');

    // Configure token mapping so that the module attempts to save values back
    // to Recurly account objects.
    $this->config('recurly.settings')
      ->set('recurly_token_mapping', [
        'email' => '[user:mail]',
        'username' => '[user:name]',
      ])
      ->save();

    // Add a user and Recurly subscription, this matches what's in the API
    // fixtures.
    $user = $this->createUser();
    $account_code = 'abcdef1234567890';
    recurly_account_save_local(['code' => $account_code], 'user', $user->id());
    $this->user = $user;
  }

  /**
   * Tests that errors are logged if the Recurly account can't be loaded.
   *
   * This replicates what would happen if trying to load the account from the
   * Recurly API resulted in a NotFound or other error.
   *
   * @covers \Drupal\recurly\RecurlyEntityOperations::entityUpdate
   */
  public function testEntityUpdate() {
    $client = $this->prophesize(Client::class);
    $client->getAccount(Argument::type('string'))
      ->willThrow(new NotFound('Mock not found'));

    $this->setupMockRecurlyClient($client->reveal());

    $messenger = $this->prophesize(MessengerInterface::class);
    $this->container->set('messenger', $messenger->reveal());

    // Then update the user, and verify the entityUpdate method is triggered.
    $old_email = $this->user->mail;
    $new_email = $this->randomMachineName() . '@example.com';
    $this->assertNotEquals($new_email, $old_email);
    $this->user->mail = $new_email;

    // This should trigger RecurlyEntityOperations::entityUpdate, which should
    // make a call to the Recurly API. First, we should get an error if the
    // Recurly_Account object can't be loaded.
    $this->user->save();

    // Check that our message was added to the messenger service. Gets called
    // 2 times, once for the RecurlyMockClient notices.
    $messenger->addWarning(Argument::that(function ($arg) {
      return stristr($arg, 'Unable to save updated account data') || stristr($arg, 'RecurlyMockClient');
    }))->shouldHaveBeenCalledTimes(1);
  }

  /**
   * Tests that errors are logged if the account data can't be saved to Recurly.
   *
   * @covers \Drupal\recurly\RecurlyEntityOperations::entityUpdate
   */
  public function testEntityUpdatePutRequestError() {
    $client = $this->prophesize(Client::class);
    $client->getAccount(Argument::type('string'))
      ->willReturn($this->getMockAccount());

    // Mock a failed attempt to update the account via the Recurly API.
    $client->updateAccount(Argument::type('string'), Argument::cetera())
      ->willThrow(new RecurlyError('Mock error'));

    $this->setupMockRecurlyClient($client->reveal());

    $messenger = $this->prophesize(MessengerInterface::class);
    $this->container->set('messenger', $messenger->reveal());

    // This should get an error when the PUT request to update the account
    // fails.
    $old_email = $this->user->mail;
    $new_email = $this->randomMachineName() . '@example.com';
    $this->assertNotEquals($new_email, $old_email);
    $this->user->mail = $new_email;
    $this->user->save();

    $client->getAccount(Argument::type('string'))
      ->shouldHaveBeenCalledTimes(1);
    $messenger->addWarning(Argument::that(function ($arg) {
      return stristr($arg, 'The billing system reported an error') || stristr($arg, 'RecurlyMockClient');
    }))->shouldHaveBeenCalledTimes(1);
  }

  /**
   * Tests that data can succsesfully be safed to Recurly.
   *
   * @covers \Drupal\recurly\RecurlyEntityOperations::entityUpdate
   */
  public function testEntityUpdateSucess() {
    $client = $this->prophesize(Client::class);

    // Set up a mock response for getting the account from Recurly.
    $recurly_account = $this->getMockAccount();
    $client->getAccount(Argument::type('string'))
      ->willReturn($recurly_account);

    // Mock a successful attempt to update the account via the Recurly API.
    $client->updateAccount(Argument::type('string'), Argument::cetera())
      ->willReturn($recurly_account);

    $this->setupMockRecurlyClient($client->reveal());

    $messenger = $this->prophesize(MessengerInterface::class);
    $this->container->set('messenger', $messenger->reveal());

    $old_email = $this->user->mail;
    $new_email = $this->randomMachineName() . '@example.com';
    $this->assertNotEquals($new_email, $old_email);
    $this->user->mail = $new_email;
    $this->user->save();

    $client->getAccount(Argument::type('string'))
      ->shouldHaveBeenCalledTimes(1);
    $client->updateAccount(Argument::type('string'), Argument::cetera())
      ->shouldHaveBeenCalledTimes(1);
    $messenger->addWarning(Argument::cetera())
      ->shouldNotBeCalled();
  }

}
