<?php

namespace Drupal\Tests\recurly\Kernel;

use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\recurly\Traits\RecurlyTestTrait;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\recurly\Controller\RecurlyInvoicesController;
use Drupal\recurly_test_client\RecurlyV3MockClient;
use Prophecy\Argument;
use Prophecy\PhpUnit\ProphecyTrait;
use Recurly\Client;
use Recurly\Errors\NotFound;
use Recurly\Pager;
use Recurly\Resources\BinaryFile;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Tests for RecurlyInvoicesControllerTest.
 *
 * @covers \Drupal\recurly\Controller\RecurlyInvoicesControllerTest
 * @group recurly
 */
class RecurlyInvoicesControllerTest extends KernelTestBase {

  use ProphecyTrait;
  use RecurlyTestTrait;
  use UserCreationTrait;

  /**
   * Instance of RecurlyInvoicesControllerTest.
   *
   * @var \Drupal\recurly\Controller\RecurlyController
   */
  protected $controller;

  /**
   * Drupal user object.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $drupalUser;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'recurly',
    'system',
    'user',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp() : void {
    parent::setUp();

    $this->installConfig(['recurly', 'user', 'system']);
    $this->config('recurly.settings')
      ->set('recurly_entity_type', 'user')
      ->set('recurly_subscription_plans',
        ['silver' => ['status' => '1', 'weight' => '0']])
      ->save();

    $this->rebuildRecurlyEntityRouting();

    $this->installSchema('recurly', ['recurly_account']);
    $this->installEntitySchema('user');

    // Add a user and Recurly subscription, this matches what's in the API
    // fixtures.
    $user = $this->setUpCurrentUser();
    $account_code = 'abcdef1234567890';
    recurly_account_save_local(['code' => $account_code], 'user', $user->id());

    $this->drupalUser = $user;
  }

  /**
   * Tests for invoice list UI.
   *
   * @coverrs \Drupal\recurly\Controller\RecurlyInvoicesController::invoicesList
   */
  public function testInvoicesList() {
    $client = $this->prophesize(Client::class);
    // Mock the call to GET /accounts/{account_id}.
    $client->getAccount(Argument::type('string'))
      ->willReturn($this->getMockAccount(['code' => 'abcdef1234567890']));

    // Mock the call to GET /accounts/{account_id}/invoices.
    $list = RecurlyV3MockClient::createMockRecurlyResponse(
      RecurlyV3MockClient::loadRecurlyJsonFixture('invoices/index-200'),
      ['HTTP/1.1 200 OK', 'Recurly-Total-Records: 2']
    );
    $client->pagerCount(Argument::cetera())
      ->willReturn($list);
    $client->nextPage(Argument::cetera())
      ->willReturn($list->toResource());

    // Mock the paged Recurly response for the invoices list.
    $pager = new Pager($client->reveal(), 'page_one');
    $client->listAccountInvoices(Argument::cetera())
      ->willReturn($pager);

    // This is required for recurly_load_account().
    $this->setupMockRecurlyClient($client->reveal());

    $controller = new RecurlyInvoicesController(
      $this->container->get('recurly.pager_manager'),
      $this->container->get('recurly.format_manager'),
      $this->mockClientFactory->reveal()
    );

    $routeMatch = $this->prophesize(RouteMatchInterface::class);
    $routeMatch->getParameter('user')->willReturn($this->drupalUser);

    $response = $controller->invoicesList($routeMatch->reveal());
    $this->assertArrayHasKey('#theme', $response);
    $this->assertArrayHasKey('#invoices', $response);
    $this->assertEquals(count($response['#invoices']), $response['#total']);
  }

  /**
   * Tests for display of individual invoice.
   *
   * @covers \Drupal\recurly\Controller\RecurlyInvoicesController::getInvoice
   */
  public function testGetInvoice() {
    $client = $this->prophesize(Client::class);
    // Mock the call to GET /accounts/{account_id}.
    $client->getAccount(Argument::type('string'))
      // Set the account code to match what is in the show-200.json fixture.
      ->willReturn($this->getMockAccount(['code' => 'abcdef1234567890']));

    $invoice = RecurlyV3MockClient::createMockRecurlyResponse(RecurlyV3MockClient::loadRecurlyJsonFixture('invoices/show-200'));
    /** @var \Recurly\Resources\Invoice $invoice_resource */
    $invoice_resource = $invoice->toResource();
    $client->getInvoice(Argument::type('string'))
      ->willReturn($invoice_resource);

    // Mock route for the controller.
    $routeMatch = $this->prophesize(RouteMatchInterface::class);
    $routeMatch->getParameter('user')->willReturn($this->drupalUser);

    $this->setupMockRecurlyClient($client->reveal());

    $controller = new RecurlyInvoicesController(
      $this->container->get('recurly.pager_manager'),
      $this->container->get('recurly.format_manager'),
      $this->mockClientFactory->reveal()
    );

    $response = $controller->getInvoice($routeMatch->reveal(), '1000');
    $this->assertEquals('recurly_invoice', $response['#theme']);
    $this->assertArrayHasKey('#invoice', $response);
    $this->assertEquals(NULL, $response['#error_message']);

    // Verify #error_message if invoice state is not 'paid'.
    $invoice_resource->setState('past_due');
    $client->getInvoice(Argument::type('string'))
      ->willReturn($invoice_resource);

    $response = $controller->getInvoice($routeMatch->reveal(), '1001');
    $this->assertEquals('recurly_invoice', $response['#theme']);
    $this->assertStringContainsString('This invoice is past due!', $response['#error_message']);

    // Can't find invoice with that ID.
    $client->getInvoice(Argument::type('string'))
      ->willThrow(new NotFound('Invoice not found'));
    $this->expectException(NotFoundHttpException::class);
    $this->expectExceptionMessage('Invoice not found');
    $controller->getInvoice($routeMatch->reveal(), 'bad-invoice-id');

    // The user identified by the route does not match the user associated with
    // the invoice ID.
    $client->getInvoice(Argument::type('string'))
      ->willReturn($invoice_resource);
    $alternateUser = $this->createUser();
    $routeMatch->getParameter('user')->willReturn($alternateUser);
    $this->expectException(NotFoundHttpException::class);
    $this->expectExceptionMessage('User account does not match invoice account');
    $controller->getInvoice($routeMatch->reveal(), '1000');
  }

  /**
   * Tests for invoice PDF feature.
   *
   * @covers \Drupal\recurly\Controller\RecurlyInvoicesController::getInvoicePdf
   */
  public function testGetPdf() {
    $client = $this->prophesize(Client::class);
    // Mock the call to GET /accounts/{account_id}.
    $client->getAccount(Argument::type('string'))
      // Set the account code to match what is in the show-200.json fixture.
      ->willReturn($this->getMockAccount(['code' => 'abcdef1234567890']));

    $invoice = RecurlyV3MockClient::createMockRecurlyResponse(RecurlyV3MockClient::loadRecurlyJsonFixture('invoices/show-200'));
    /** @var \Recurly\Resources\Invoice $invoice_resource */
    $invoice_resource = $invoice->toResource();
    $client->getInvoice(Argument::type('string'))
      ->willReturn($invoice_resource);

    $pdf = new BinaryFile();
    $client->getInvoicePdf(Argument::type('string'))
      ->willReturn($pdf);

    $this->setupMockRecurlyClient($client->reveal());

    // Mock the route for the controller.
    $routeMatch = $this->prophesize(RouteMatchInterface::class);
    $routeMatch->getParameter('user')->willReturn($this->drupalUser);

    $controller = new RecurlyInvoicesController(
      $this->container->get('recurly.pager_manager'),
      $this->container->get('recurly.format_manager'),
      $this->mockClientFactory->reveal()
    );

    $response = $controller->getInvoicePdf($routeMatch->reveal(), '1000');

    $this->assertInstanceOf(Response::class, $response);
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertEquals('application/pdf', $response->headers->get('content-type'));

    // Can't find invoice with that ID.
    $this->expectException(NotFoundHttpException::class);
    $this->expectExceptionMessage('Invoice not found');
    $controller->getInvoicePdf($routeMatch->reveal(), 'bad-invoice-id');

    // The user identified by the route does not match the user associated with
    // the invoice ID.
    $alternateUser = $this->createUser();
    $routeMatch->getParameter('user')->willReturn($alternateUser);
    $this->expectException(NotFoundHttpException::class);
    $this->expectExceptionMessage('User account does not match invoice account');
    $controller->getInvoicePdf($routeMatch->reveal(), '1000');
  }

}
