<?php

namespace Drupal\Tests\recurly\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\recurly\Traits\RecurlyTestTrait;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\recurly\Controller\RecurlyManageSubscriptionController;
use Prophecy\Argument;
use Recurly\Client;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Tests for RecurlyManageSubscriptionController.
 *
 * @covers \Drupal\recurly\Controller\RecurlyManageSubscriptionController
 * @group recurly
 */
class RecurlyManageSubscriptionControllerTest extends KernelTestBase {

  use RecurlyTestTrait;
  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'recurly',
    'system',
    'user',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp() : void {
    parent::setUp();

    $this->installConfig(['recurly', 'user', 'system']);
    $this->config('recurly.settings')
      ->set('recurly_entity_type', 'user')
      ->set('recurly_subscription_plans', [
        'silver' => [
          'status' => '1',
          'weight' => '0',
        ],
      ])
      ->save();

    $this->rebuildRecurlyEntityRouting();

    $this->installSchema('recurly', ['recurly_account']);
    $this->installEntitySchema('user');
  }

  /**
   * Tests that a redirect is returned for valid account codes.
   *
   * @covers \Drupal\recurly\Controller\RecurlyManageSubscriptionController::subscriptionRedirect
   */
  public function testSubscriptionRedirect() {
    $user = $this->setUpCurrentUser();
    // Add a Recurly subscription to the user.
    $account_code = 'abcdef1234567890';
    recurly_account_save_local(['code' => $account_code], 'user', $user->id());

    // Mock the API request for the account from recurly_load_account().
    $client = $this->prophesize(Client::class);
    // Mock the call to GET /accounts/{account_id}.
    $client->getAccount(Argument::type('string'))
      ->willReturn($this->getMockAccount(['code' => 'abcdef1234567890']));

    $this->setupMockRecurlyClient($client->reveal());

    $controller = new RecurlyManageSubscriptionController(
      $this->container->get('recurly.pager_manager'),
      $this->container->get('recurly.format_manager'),
      $this->mockClientFactory->reveal()
    );

    $response = $controller->subscriptionRedirect($account_code);
    $this->assertInstanceOf('\Symfony\Component\HttpFoundation\RedirectResponse', $response);
    $this->assertEquals(302, $response->getStatusCode());
    $this->assertStringContainsString('/user/' . $user->id() . '/subscription', $response->getTargetUrl());

    $this->expectException(NotFoundHttpException::class);
    $controller->subscriptionRedirect('bad-account-code');
  }

}
