<?php

namespace Drupal\Tests\recurly\Kernel;

use Drupal\Core\Pager\PagerManagerInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\recurly\Traits\RecurlyTestTrait;
use Drupal\recurly\RecurlyPagerManager;
use Drupal\recurly_test_client\RecurlyV3MockClient;
use Prophecy\Argument;
use Prophecy\PhpUnit\ProphecyTrait;
use Recurly\Client;
use Recurly\Pager;

/**
 * Tests for RecurlyPagerManager.
 *
 * @covers \Drupal\recurly\RecurlyPagerManager
 * @group recurly
 */
class RecurlyPagerManagerTest extends KernelTestBase {

  use ProphecyTrait;
  use RecurlyTestTrait;

  /**
   * Recurly pager object that mocks some features.
   *
   * @var \Recurly\Pager
   */
  protected $mockRecurlyPager;

  /**
   * Count.
   *
   * @var int
   */
  protected $count = 3;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['recurly', 'system'];

  /**
   * {@inheritdoc}
   */
  protected function setUp() : void {
    parent::setUp();
    $client = $this->prophesize(Client::class);

    $pager_count_response = RecurlyV3MockClient::createMockRecurlyResponse(json_encode(['object' => 'account']));
    $pager_count_response->setHeaders([
      'HTTP/1.1 200 OK',
      "Recurly-Total-Records: {$this->count}",
    ]);
    $client->pagerCount(Argument::cetera())
      ->willReturn($pager_count_response);

    $json = [
      "object" => "list",
      "has_more" => TRUE,
      "next" => "page_two",
      "data" => [
        [
          "object" => "account",
          "id" => "resource one",
        ],
        [
          "object" => "account",
          "id" => "resource two",
        ],
        [
          "object" => "account",
          "id" => "resource three",
        ],
      ],
    ];
    $next_page_response = RecurlyV3MockClient::createMockRecurlyResponse(json_encode($json));
    $next_page_response->setHeaders([
      'HTTP/1.1 200 OK',
      "Recurly-Total-Records: {$this->count}",
    ]);
    $client->nextPage(Argument::cetera())
      ->willReturn($next_page_response->toResource());

    $this->mockRecurlyPager = new Pager($client->reveal(), 'page_one');
  }

  /**
   * Tests for pager service.
   *
   * @covers \Drupal\recurly\RecurlyPagerManager::pagerResults
   */
  public function testPagerResults() {
    // Mock the Drupal pager service. And ensure that it's called from within
    // the recurly pager service to initialize a pager for theming.
    $mockDrupalPagerService = $this->prophesize(PagerManagerInterface::class);
    $mockDrupalPagerService->findPage()
      ->willReturn(0)
      ->shouldBeCalled();
    $mockDrupalPagerService->createPager(
        Argument::is("3"),
        Argument::is(2)
      )
      ->shouldBeCalled();

    $pager = new RecurlyPagerManager($mockDrupalPagerService->reveal());

    $per_page = 2;
    $result = $pager->pagerResults($this->mockRecurlyPager, $per_page);

    // Verify we got the desired items.
    $this->assertCount($per_page, $result);
    // And that it's the first 2 items.
    $this->assertArrayHasKey('resource one', $result);
    $this->assertArrayHasKey('resource two', $result);

    // Advance the pager.
    $result = $pager->pagerResults($this->mockRecurlyPager, $per_page, 1);
    // The 2nd page only has 1 item.
    $this->assertCount(1, $result);
    // Verify that it's the second 5 items.
    $this->assertArrayHasKey('resource three', $result);
  }

}
