<?php

namespace Drupal\Tests\recurly\Kernel;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\recurly\Traits\RecurlyTestTrait;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\recurly\Controller\RecurlyPushListenerController;
use Prophecy\Argument;
use Prophecy\PhpUnit\ProphecyTrait;
use Recurly\Client;
use Recurly\Resources\AccountMini;
use Recurly\Resources\Subscription;
use Symfony\Component\HttpFoundation\Request;

/**
 * Tests for RecurlyPushListenerController.
 *
 * @covers \Drupal\recurly\Controller\RecurlyPushListenerController
 * @group recurly
 */
class RecurlyPushListenerControllerTest extends KernelTestBase {

  use ProphecyTrait;
  use RecurlyTestTrait;
  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'recurly',
    'recurly_test_client',
    'system',
    'user',
  ];

  /**
   * Drupal user object.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $drupalUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp() : void {
    parent::setUp();

    $this->installConfig(['recurly', 'user', 'system']);

    $this->config('recurly.settings')
      ->set('recurly_entity_type', 'user')
      ->set('recurly_subscription_plans',
        ['silver' => ['status' => '1', 'weight' => '0']])
      ->save();

    $this->rebuildRecurlyEntityRouting();

    $this->installSchema('recurly', ['recurly_account']);
    $this->installEntitySchema('user');
  }

  /**
   * Test handling of empty or invalid push notifications.
   */
  public function testEmptyNotification() {
    $payload = '{}';
    $request = $this->prophesize(Request::class);
    $request->getContentTypeFormat()->willReturn('json');
    $request->getContent()->willReturn($payload);

    /** @var \Drupal\recurly\Controller\RecurlyPushListenerController $controller */
    $controller = RecurlyPushListenerController::create($this->container);
    $response = $controller->processPushNotification($request->reveal(), NULL);
    $this->assertEquals(400, $response->getStatusCode());
    $this->assertStringContainsString('Empty or invalid notification.', $response->getContent());
  }

  /**
   * Test validation of the push notification unique key parameter.
   */
  public function testKeyValidation() {
    $key = 'asdf1234';
    $invalid_key = 'jklm7890';
    $this->config('recurly.settings')
      ->set('recurly_listener_key', $key)
      ->save();

    // For when the Recurly account gets loaded.
    // Mock the API request for the account from recurly_load_account().
    $client = $this->prophesize(Client::class);
    // Mock the call to GET /accounts/{account_id}.
    $client->getAccount(Argument::type('string'))
      ->willReturn($this->getMockAccount(['code' => 'abcdef1234567890', 'email' => 'abcedf1234567890@example.com']));

    $this->setupMockRecurlyClient($client->reveal());

    // Note that the account_code here matches what's in mock above.
    $payload = '{"id": "r3oplsj3zo7a","object_type": "account","site_id": "r23khg9b5kyb","event_type": "created","event_time": "2022-06-24T19:55:25Z", "account_code": "user-abcdef1234567890"}';

    $request = $this->prophesize(Request::class);
    $request->getContentTypeFormat()->willReturn('json');
    $request->getContent()->willReturn($payload);

    /** @var \Drupal\recurly\Controller\RecurlyPushListenerController $controller */
    $controller = RecurlyPushListenerController::create($this->container);

    // Invalid key.
    $response = $controller->processPushNotification($request->reveal(), $invalid_key, NULL);
    $this->assertNotEquals($key, $invalid_key);
    $this->assertStringContainsString('Incoming push notification did not contain the proper URL key.', $response->getContent());

    // Valid key.
    $response = $controller->processPushNotification($request->reveal(), $key, NULL);
    $this->assertStringNotContainsString('Incoming push notification did not contain the proper URL key.', $response->getContent());
  }

  /**
   * Test handling of logic for finding an account_code in the notification.
   */
  public function testAccountCodeValidation() {
    // Code to test the logic that finds an account_code in the notification,
    // tries to find one by loading the object the notification refers too if
    // needed, and then displays a message if it doesn't find one.
    //
    // For when the Recurly account gets loaded.
    // Mock the API request for the account from recurly_load_account().
    $client = $this->prophesize(Client::class);
    // Mock the call to GET /accounts/{account_id}.
    $client->getAccount(Argument::type('string'))
      ->willReturn($this->getMockAccount(['code' => 'abcdef1234567890', 'email' => 'abcedf1234567890@example.com']));
    $this->setupMockRecurlyClient($client->reveal());

    $request = $this->prophesize(Request::class);
    $request->getContentTypeFormat()->willReturn('json');

    // First, test that if there is no account_code in the notification, the
    // controller should return an error message. This payload is purposely
    // malformed because it's missing the account_code field.
    $payload = '{"object_type": "account", "event_type": "created"}';
    $request->getContent()->willReturn($payload);

    /** @var \Drupal\recurly\Controller\RecurlyPushListenerController $controller */
    $controller = RecurlyPushListenerController::create($this->container);

    $response = $controller->processPushNotification($request->reveal(), NULL);
    $this->assertStringContainsString('Unable to retrieve Recurly account associated with push notification.', $response->getContent());

    // Second, test that if there is an account_code in the notification, the
    // controller should try to load the account. And not result in an error.
    $payload = '{"object_type": "account","event_type": "created", "account_code": "user-abcdef1234567890"}';
    $request->getContent()->willReturn($payload);
    $response = $controller->processPushNotification($request->reveal(), NULL);
    $client->getAccount(Argument::type('string'))->shouldHaveBeenCalled();
    $this->assertStringNotContainsString('Recurly account could not be loaded from API', $response->getContent());

    // Finally, test that if the notification is for a subscription, that the
    // subscription is loaded and the account code is extracted from that.
    $payload = '{"id": "r3oplsj3zo7a","object_type": "subscription","event_type": "created"}';
    $request->getContent()->willReturn($payload);
    $account = $this->prophesize(AccountMini::class);
    $account->getCode()->willReturn('user-abcdef1234567890');
    $subscription = $this->prophesize(Subscription::class);
    $subscription->getAccount()->willReturn($account->reveal());
    $client->getSubscription(Argument::type('string'))
      ->willReturn($subscription->reveal());

    $response = $controller->processPushNotification($request->reveal(), NULL);
    $client->getSubscription(Argument::type('string'))->shouldHaveBeenCalled();
    $subscription->getAccount()->shouldHaveBeenCalled();
    $account->getCode()->shouldHaveBeenCalled();
    $this->assertStringNotContainsString('Recurly account could not be loaded from API', $response->getContent());
  }

  /**
   * Tests that hooks are invoked to notify other modules of push notifications.
   */
  public function testHooksInvoked() {
    $moduleHandler = $this->prophesize(ModuleHandlerInterface::class);
    $this->container->set('module_handler', $moduleHandler->reveal());

    // Mock a valid notification payload, and data for call to load the account
    // from the Recurly API.
    $client = $this->prophesize(Client::class);
    // Mock the call to GET /accounts/{account_id}.
    $client->getAccount(Argument::type('string'))
      ->willReturn($this->getMockAccount(['code' => 'abcdef1234567890', 'email' => 'abcedf1234567890@example.com']));
    $this->setupMockRecurlyClient($client->reveal());

    $payload = '{"object_type": "account","event_type": "created", "account_code": "user-abcdef1234567890"}';
    $request = $this->prophesize(Request::class);
    $request->getContentTypeFormat()->willReturn('json');
    $request->getContent()->willReturn($payload);

    $controller = RecurlyPushListenerController::create($this->container);

    $controller->processPushNotification($request->reveal(), NULL, NULL);
    $moduleHandler->invokeAll('recurly_process_push_notification', Argument::any())->shouldHaveBeenCalled();
  }

  /**
   * Test creation of local account database record.
   *
   * When we get a push notification from recurly for a Recurly account and
   * there is no corresponding local account record we create one.
   */
  public function testLocalAccountCreation() {
    // Mock a valid notification payload, and data for call to load the account
    // from the Recurly API. This account from Recurly does not match with a
    // local account. There is no user entity with the ID 42.
    $client = $this->prophesize(Client::class);
    // Mock the call to GET /accounts/{account_id}.
    $client->getAccount(Argument::type('string'))
      ->willReturn($this->getMockAccount(['code' => 'user-42', 'email' => 'user-42@example.com']));
    $this->setupMockRecurlyClient($client->reveal());

    $payload = '{"object_type": "account","event_type": "created","account_code": "user-42"}';
    $request = $this->prophesize(Request::class);
    $request->getContentTypeFormat()->willReturn('json');
    $request->getContent()->willReturn($payload);

    $local_account = recurly_account_load(['account_code' => 'user-42'], TRUE);
    $this->assertFalse($local_account);

    $controller = RecurlyPushListenerController::create($this->container);
    $response = $controller->processPushNotification($request->reveal(), NULL);
    $this->assertEquals(200, $response->getStatusCode());
    $local_account = recurly_account_load(['account_code' => 'user-42'], TRUE);
    $this->assertEquals('user-42', $local_account->account_code);
  }

}
