<?php

namespace Drupal\Tests\recurly\Kernel;

use Drupal\Core\Render\Element;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\recurly\Traits\RecurlyTestTrait;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\recurly\Controller\RecurlySubscriptionListController;
use Drupal\recurly_test_client\RecurlyV3MockClient;
use Prophecy\Argument;
use Prophecy\PhpUnit\ProphecyTrait;
use Recurly\Client;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Tests for RecurlySubscriptionListController.
 *
 * @group recurly
 */
class RecurlySubscriptionListControllerTest extends KernelTestBase {

  use ProphecyTrait;
  use RecurlyTestTrait;
  use UserCreationTrait;

  /**
   * Drupal user object.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $drupalUser;

  /**
   * Mock subscription 1.
   *
   * @var array
   */
  protected $subscription1 = [];

  /**
   * Mock subscription 2.
   *
   * @var array
   */
  protected $subscription2 = [];

  /**
   * Mock subscription 3.
   *
   * @var array
   */
  protected $subscription3 = [];

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'recurly',
    'system',
    'user',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp() : void {
    parent::setUp();

    $this->installConfig(['recurly', 'user', 'system']);

    $this->config('recurly.settings')
      ->set('recurly_private_api_key', 'test')
      ->set('recurly_subdomain', 'test')
      ->set('recurly_entity_type', 'user')
      ->set('recurly_subscription_plans',
        ['silver' => ['status' => '1', 'weight' => '0']])
      ->save();

    $this->rebuildRecurlyEntityRouting();

    $this->installSchema('recurly', ['recurly_account']);
    $this->installEntitySchema('user');

    // Add a user and Recurly subscription, this matches what's in the API
    // fixtures.
    $user = $this->setUpCurrentUser();
    $account_code = 'abcdef1234567890';
    recurly_account_save_local(['code' => $account_code], 'user', $user->id());

    $this->subscription1 = json_decode(RecurlyV3MockClient::loadRecurlyJsonFixture('subscriptions/show-200'));
    $this->subscription2 = json_decode(RecurlyV3MockClient::loadRecurlyJsonFixture('subscriptions/show-200'));
    $this->subscription2->id = 'sub_67890';
    $this->subscription2->uuid = 'sub-67890-uuid';
    $this->subscription2->state = 'past_due';
    $this->subscription3 = json_decode(RecurlyV3MockClient::loadRecurlyJsonFixture('subscriptions/show-200'));
    $this->subscription3->id = 'sub_asdf';
    $this->subscription3->uuid = 'sub-asdf-uuid';

    $this->drupalUser = $user;
  }

  /**
   * Test configuration settings for subscription list page.
   *
   * @covers \Drupal\recurly\Controller\RecurlySubscriptionListController::subscriptionList
   */
  public function testSubscriptionList() {
    // For when the Recurly account gets loaded.
    // Mock the API request for the account from recurly_load_account().
    $client = $this->prophesize(Client::class);
    // Mock the call to GET /accounts/{account_id}.
    $client->getAccount(Argument::type('string'))
      ->willReturn($this->getMockAccount(['code' => 'abcdef1234567890']));

    // Building a list of subscriptions makes an API call for a list of
    // coupon redemptions, and past_due invoices. For this test, we'll just
    // return an empty result for both of those.
    //
    // First coupons.
    $empty_data = [
      'object' => 'list',
      'has_more' => FALSE,
      'next' => NULL,
      'data' => [],
    ];
    $pager = $this->createRecurlyPagedResponse($empty_data, 'coupons/test/path');
    $client->listActiveCouponRedemptions(Argument::cetera())
      ->willReturn($pager);

    // And then invoices.
    $pager = $this->createRecurlyPagedResponse($empty_data, 'invoices/test/path');
    $client->listAccountInvoices(Argument::type('string'), Argument::exact(['params' => ['state' => 'past_due']]))
      ->willReturn($pager)
      ->shouldBeCalledTimes(1);

    // Configure the module to only list 'live' subscriptions.
    $this->config('recurly.settings')
      ->set('recurly_subscription_display', 'live')
      ->save();

    // Return 1 live subscription.
    $data_live = [
      'object' => 'list',
      'has_more' => FALSE,
      'next' => NULL,
      'data' => [
        $this->subscription1,
      ],
    ];
    $pager = $this->createRecurlyPagedResponse($data_live, 'subscription/test/live');
    $client->listAccountSubscriptions(Argument::type('string'), Argument::exact(['params' => ['state' => 'live']]))
      ->willReturn($pager);

    $routeMatch = $this->prophesize(RouteMatchInterface::class);
    $routeMatch->getParameter('user')->willReturn($this->drupalUser);

    $this->setupMockRecurlyClient($client->reveal());

    $controller = new RecurlySubscriptionListController(
      $this->container->get('recurly.pager_manager'),
      $this->container->get('recurly.format_manager'),
      $this->mockClientFactory->reveal(),
      $this->container->get('datetime.time'),
    );

    $response = $controller->subscriptionList($routeMatch->reveal());

    // Verifies that the API is called with the 'live' parameter.
    $this->assertCount(1, Element::children($response['subscriptions']));
    $client->listAccountSubscriptions(Argument::type('string'), Argument::exact(['params' => ['state' => 'live']]))
      ->shouldHaveBeenCalledTimes(1);

    // Add two more subscriptions to the list, and reset the response so that
    // the next request will return 3 results instead of 1. Verifies that the
    // controller can handle listing multiple subscriptions.
    $data_all = $data_live;
    $data_all['data'][] = $this->subscription2;
    $data_all['data'][] = $this->subscription3;
    $pager = $this->createRecurlyPagedResponse($data_all, 'subscription/test/all');
    $client->listAccountSubscriptions(Argument::type('string'), Argument::exact(['params' => ['state' => 'live']]))
      ->willReturn($pager);
    $response = $controller->subscriptionList($routeMatch->reveal());
    $this->assertCount(3, Element::children($response['subscriptions']));

    // List 'all' subscriptions.
    $this->config('recurly.settings')
      ->set('recurly_subscription_display', 'all')
      ->save();

    // When 'all' subscriptions are displayed the state=active param is not
    // included in the API request.
    $new_pager = $this->createRecurlyPagedResponse($data_all, 'subscription/test/active');
    $client->listAccountSubscriptions(Argument::type('string'), Argument::exact(['params' => []]))
      ->willReturn($new_pager)
      ->shouldBeCalledTimes(1);

    $controller->subscriptionList($routeMatch->reveal());
  }

  /**
   * Verify user is redirected if they do not have a recurly account.
   *
   * @covers \Drupal\recurly\Controller\RecurlySubscriptionListController::subscriptionList
   */
  public function testNoAccountBehavior() {
    $account = $this->createUser(['manage recurly subscription']);
    $routeMatch = $this->prophesize(RouteMatchInterface::class);
    $routeMatch->getParameter('user')->willReturn($account);

    // User does not have a recurly account.
    $recurly_account = recurly_account_load([
      'entity_type' => 'user',
      'entity_id' => $account->id(),
    ]);
    $this->assertFalse($recurly_account);

    // No need to mock the API client because this path should not make any
    // API calls.
    $controller = new RecurlySubscriptionListController(
      $this->container->get('recurly.pager_manager'),
      $this->container->get('recurly.format_manager'),
      $this->container->get('recurly.client'),
      $this->container->get('datetime.time'),
    );

    $response = $controller->subscriptionList($routeMatch->reveal());
    $this->assertInstanceOf(RedirectResponse::class, $response);
  }

  /**
   * Test user is redirected if they have no subscriptions.
   *
   * @covers \Drupal\recurly\Controller\RecurlySubscriptionListController::subscriptionList
   */
  public function testNoSubscriptionsBehavior() {
    $client = $this->prophesize(Client::class);
    // Mock the call to GET /accounts/{account_id}.
    $client->getAccount(Argument::type('string'))
      ->willReturn($this->getMockAccount(['code' => 'abcdef1234567890']));

    // Return empty subscription list.
    $data = [
      'object' => 'list',
      'has_more' => FALSE,
      'next' => NULL,
      'data' => [],
    ];
    $pager = $this->createRecurlyPagedResponse($data, 'subscriptions/test/list');
    $client->listAccountSubscriptions(Argument::type('string'), Argument::exact(['params' => ['state' => 'live']]))
      ->willReturn($pager);

    $coupon_pager = $this->createRecurlyPagedResponse($data, 'coupons/test/list');
    $client->listActiveCouponRedemptions(Argument::cetera())
      ->willReturn($coupon_pager);

    $routeMatch = $this->prophesize(RouteMatchInterface::class);
    $routeMatch->getParameter('user')->willReturn($this->drupalUser);

    $this->setupMockRecurlyClient($client->reveal());

    $controller = new RecurlySubscriptionListController(
      $this->container->get('recurly.pager_manager'),
      $this->container->get('recurly.format_manager'),
      $this->mockClientFactory->reveal(),
      $this->container->get('datetime.time'),
    );

    $response = $controller->subscriptionList($routeMatch->reveal());
    $this->assertInstanceOf(RedirectResponse::class, $response);
  }

  /**
   * Tests calculation of subscription states.
   *
   * @covers \Drupal\recurly\Controller\RecurlySubscriptionListController::subscriptionGetStates
   */
  public function testSubscriptionListStates() {
    $client = $this->prophesize(Client::class);
    // Mock the call to GET /accounts/{account_id}.
    $client->getAccount(Argument::type('string'))
      ->willReturn($this->getMockAccount(['code' => 'abcdef1234567890']));

    // Empty coupon list.
    $empty_data = [
      'object' => 'list',
      'has_more' => FALSE,
      'next' => NULL,
      'data' => [],
    ];
    $pager = $this->createRecurlyPagedResponse($empty_data, 'coupons/test/path');
    $client->listActiveCouponRedemptions(Argument::cetera())
      ->willReturn($pager);

    // Empty invoices list.
    $pager = $this->createRecurlyPagedResponse($empty_data, 'invoices/test/path');
    $client->listAccountInvoices(Argument::type('string'), Argument::exact(['params' => ['state' => 'past_due']]))
      ->willReturn($pager);

    $this->setupMockRecurlyClient($client->reveal());

    $routeMatch = $this->prophesize(RouteMatchInterface::class);
    $routeMatch->getParameter('user')->willReturn($this->drupalUser);

    $controller = new RecurlySubscriptionListController(
      $this->container->get('recurly.pager_manager'),
      $this->container->get('recurly.format_manager'),
      $this->mockClientFactory->reveal(),
      $this->container->get('datetime.time'),
    );

    // These are the base states that a subscription can be in as stored in the
    // Recurly API. Other states like 'past_due' are calculated states.
    $base_states = ['active', 'canceled', 'expired', 'pending', 'future'];
    foreach ($base_states as $state) {
      $subscription = $this->subscription1;
      $subscription->state = $state;
      $data_live = [
        'object' => 'list',
        'has_more' => FALSE,
        'next' => NULL,
        'data' => [
          $subscription,
        ],
      ];
      $pager = $this->createRecurlyPagedResponse($data_live, 'subscription/test/' . $state);
      $client->listAccountSubscriptions(Argument::type('string'), Argument::exact(['params' => ['state' => 'live']]))
        ->willReturn($pager);

      $response = $controller->subscriptionList($routeMatch->reveal());
      $this->assertSame([$state], $response['subscriptions'][$subscription->uuid]['#state_array']);
    }

    // The 'in_trial' state is added if the subscription has trial start/end
    // dates.
    $subscription = clone $this->subscription1;
    $subscription->state = 'active';
    $subscription->trial_started_at = date('c', strtotime('-2 weeks'));
    $subscription->trial_ends_at = date('c', strtotime('+2 weeks'));
    $data_live = [
      'object' => 'list',
      'has_more' => FALSE,
      'next' => NULL,
      'data' => [
        $subscription,
      ],
    ];
    $pager = $this->createRecurlyPagedResponse($data_live, 'subscription/test/in_trial');
    $client->listAccountSubscriptions(Argument::type('string'), Argument::exact(['params' => ['state' => 'live']]))
      ->willReturn($pager);

    $response = $controller->subscriptionList($routeMatch->reveal());
    $this->assertSame(['in_trial', 'active'], $response['subscriptions'][$subscription->uuid]['#state_array']);

    // The 'non_renewing' state is added if the subscription is set to
    // auto_renew = false.
    $subscription = clone $this->subscription1;
    $subscription->state = 'active';
    $subscription->auto_renew = FALSE;
    $data_live = [
      'object' => 'list',
      'has_more' => FALSE,
      'next' => NULL,
      'data' => [
        $subscription,
      ],
    ];
    $pager = $this->createRecurlyPagedResponse($data_live, 'subscription/test/auto_renew');
    $client->listAccountSubscriptions(Argument::type('string'), Argument::exact(['params' => ['state' => 'live']]))
      ->willReturn($pager);

    $response = $controller->subscriptionList($routeMatch->reveal());
    $this->assertSame(['non_renewing', 'active'], $response['subscriptions'][$subscription->uuid]['#state_array']);

    // The past due state is set if the subscription is associated with an
    // invoice that is past due. This requires updating the invoices list to
    // return a value other than empty.
    $subscription = clone $this->subscription1;
    $subscription->state = 'active';
    $data_live = [
      'object' => 'list',
      'has_more' => FALSE,
      'next' => NULL,
      'data' => [
        $subscription,
      ],
    ];
    $pager = $this->createRecurlyPagedResponse($data_live, 'subscription/test/past_due');
    $client->listAccountSubscriptions(Argument::type('string'), Argument::exact(['params' => ['state' => 'live']]))
      ->willReturn($pager);

    $invoice = [
      'object' => 'invoice',
      'id' => 'inv-1',
      'state' => 'past_due',
      'subscription_ids' => [$subscription->id],
    ];
    $invoice = (object) $invoice;
    $invoices_data = [
      'object' => 'list',
      'has_more' => FALSE,
      'next' => NULL,
      'data' => [
        $invoice,
      ],
    ];
    $pager = $this->createRecurlyPagedResponse($invoices_data, 'invoices/test/past_due');
    $client->listAccountInvoices(Argument::type('string'), Argument::exact(['params' => ['state' => 'past_due']]))
      ->willReturn($pager);

    $controller->clearPastDueCache();
    $response = $controller->subscriptionList($routeMatch->reveal());
    $this->assertSame(['past_due', 'active'], $response['subscriptions'][$subscription->uuid]['#state_array']);
  }

}
