<?php

namespace Drupal\Tests\recurly\Kernel;

use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\recurly\Traits\RecurlyTestTrait;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\recurly\Controller\RecurlySubscriptionSelectPlanController;
use Drupal\recurly_test_client\RecurlyV3MockClient;
use Prophecy\Argument;
use Prophecy\PhpUnit\ProphecyTrait;
use Recurly\Client;
use Symfony\Component\HttpFoundation\ParameterBag;

/**
 * Test different redirects from recurly.routing.yml.
 *
 * @covers \Drupal\recurly\Controller\RecurlySubscriptionSelectPlanController
 * @group recurly
 */
class RecurlySubscriptionSelectPlanControllerTest extends KernelTestBase {

  use ProphecyTrait;
  use RecurlyTestTrait;
  use UserCreationTrait;

  /**
   * Mock silver plan.
   *
   * @var array
   */
  protected $silverPlan = [];

  /**
   * Mock bedrock plan.
   *
   * @var array
   */
  protected $bedrockPlan = [];

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'recurly',
    'system',
    'user',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp() : void {
    parent::setUp();

    $this->installConfig(['recurly', 'user', 'system']);
    $this->config('recurly.settings')
      ->set('recurly_private_api_key', 'test')
      ->set('recurly_subdomain', 'test')
      ->set('recurly_entity_type', 'user')
      ->set('recurly_subscription_plans', [
        'silver' => [
          'status' => '1',
          'weight' => '0',
        ],
      ])
      ->save();

    $this->rebuildRecurlyEntityRouting();

    $this->installSchema('recurly', ['recurly_account']);
    $this->installEntitySchema('user');

    $silver_plan = json_decode(RecurlyV3MockClient::loadRecurlyJsonFixture('plans/show-200'));
    $this->silverPlan = $silver_plan;
    $this->bedrockPlan = clone $silver_plan;
    $this->bedrockPlan->id = 'bedrock';
    $this->bedrockPlan->code = 'bedrock';
    $this->bedrockPlan->description = 'Bedrock is at the bottom.';
  }

  /**
   * This also validates the route /subscription/register.
   *
   * @covers \Drupal\recurly\Controller\RecurlySubscriptionSelectPlanController::redirectToRegistration
   */
  public function testRegistrationRedirect() {
    $controller = RecurlySubscriptionSelectPlanController::create(\Drupal::getContainer());
    $response = $controller->redirectToRegistration();
    $this->assertInstanceOf('\Symfony\Component\HttpFoundation\RedirectResponse', $response);
    $this->assertEquals(302, $response->getStatusCode());
    $this->assertStringContainsString('/user/register', $response->getTargetUrl());
  }

  /**
   * This helps validate the route /user/signup for anon. users.
   *
   * @covers \Drupal\recurly\Controller\RecurlySubscriptionSelectPlanController::planSelect
   */
  public function testPlanSelectAnonUserSignup() {
    $client = $this->prophesize(Client::class);

    $plan_list = [
      'object' => 'list',
      'has_more' => FALSE,
      'next' => NULL,
      'data' => [
        $this->silverPlan,
        $this->bedrockPlan,
      ],
    ];
    $pager = $this->createRecurlyPagedResponse($plan_list, 'plans/test/signup');
    $client->listPlans(Argument::cetera())
      ->willReturn($pager);

    $this->setupMockRecurlyClient($client->reveal());

    // Anon users should be shown a themed plan selection widget in signup mode.
    $controller = RecurlySubscriptionSelectPlanController::create(\Drupal::getContainer());
    $response = $controller->planSelect(\Drupal::routeMatch());
    $this->assertArrayHasKey('#theme', $response);
    $this->assertEquals(RecurlySubscriptionSelectPlanController::SELECT_PLAN_MODE_SIGNUP, $response['#mode']);
  }

  /**
   * This helps validate the route /user/signup for authenticated users.
   *
   * @covers \Drupal\recurly\Controller\RecurlySubscriptionSelectPlanController::planSelect
   */
  public function testPlanSelectAuthenticatedUserSignup() {
    // Authenticated users who do not have a subscription should be redirected
    // to the authenticated user signup page.
    $controller = RecurlySubscriptionSelectPlanController::create(\Drupal::getContainer());
    $this->setUpCurrentUser();

    $response = $controller->planSelect(\Drupal::routeMatch());
    $this->assertInstanceOf('\Symfony\Component\HttpFoundation\RedirectResponse', $response);
  }

  /**
   * Plan selection for users who already have a subscription.
   *
   * @covers \Drupal\recurly\Controller\RecurlySubscriptionSelectPlanController::planSelect
   */
  public function testPlanSelectSubscriptionLookup() {
    $client = $this->prophesize(Client::class);

    $client->getAccount(Argument::type('string'))
      ->willReturn($this->getMockAccount(['code' => 'abcdef1234567890']));

    $plan_list = [
      'object' => 'list',
      'has_more' => FALSE,
      'next' => NULL,
      'data' => [
        $this->silverPlan,
        $this->bedrockPlan,
      ],
    ];
    $pager = $this->createRecurlyPagedResponse($plan_list, 'plans/test/signup');
    $client->listPlans(Argument::cetera())
      ->willReturn($pager);

    // Return subscription.
    $subscription = json_decode(RecurlyV3MockClient::loadRecurlyJsonFixture('subscriptions/show-200'));
    $data = [
      'object' => 'list',
      'has_more' => FALSE,
      'next' => NULL,
      'data' => [
        $subscription,
      ],
    ];
    $pager = $this->createRecurlyPagedResponse($data, 'subscription/test/live');
    $client->listAccountSubscriptions(Argument::type('string'))
      ->willReturn($pager);

    $this->setupMockRecurlyClient($client->reveal());

    $controller = RecurlySubscriptionSelectPlanController::create(\Drupal::getContainer());
    $user = $this->setUpCurrentUser();

    // Add a Recurly subscription to the user.
    $account_code = 'abcdef1234567890';
    recurly_account_save_local(['code' => $account_code], 'user', $user->id());

    $subscription_id = 'latest';
    $routeMatch = $this->prophesize(RouteMatchInterface::class);
    $routeMatch->getParameters()
      ->willReturn(new ParameterBag([
        'subscription_id' => $subscription_id,
        'user' => $user,
      ]));
    $routeMatch->getParameter('user')->willReturn($user);
    $response = $controller->planSelect($routeMatch->reveal(), NULL, 'latest');
    $this->assertArrayHasKey('#theme', $response);
    $this->assertEquals('change', $response['#mode']);

    // Return subscription.
    $client->getSubscription(Argument::exact($subscription->id))
      ->willReturn($this->createMockResourceFromJson(json_encode($subscription)));

    $subscription_id = $subscription->id;
    $routeMatch = $this->prophesize(RouteMatchInterface::class);
    $routeMatch->getParameters()
      ->willReturn(new ParameterBag([
        'subscription_id' => $subscription_id,
        'user' => $user,
      ]));
    $routeMatch->getParameter('user')->willReturn($user);
    $response = $controller->planSelect($routeMatch->reveal(), NULL, $subscription_id);
    $this->assertArrayHasKey('#theme', $response);
    $this->assertEquals('change', $response['#mode']);
  }

}
