<?php

declare(strict_types=1);

namespace Drupal\Tests\recurly\Traits;

use Drupal\recurly\RecurlyClientFactory;
use Drupal\recurly_test_client\RecurlyV3MockClient;
use Prophecy\Argument;
use Prophecy\PhpUnit\ProphecyTrait;
use Recurly\Client;
use Recurly\Pager;
use Recurly\RecurlyResource;
use Recurly\Request;
use Recurly\Response;

/**
 * Recurly testing helper utilities.
 */
trait RecurlyTestTrait {

  use ProphecyTrait {
    prophesize as protected recurlyProphesize;
  }


  /**
   * Recurly API client that delivers canned responses.
   *
   * @var \Drupal\recurly_test_client\RecurlyV3MockClient
   */
  protected $mockClient;

  /**
   * Mock of RecurlyClientFactory that returns a mock Recurly API client.
   *
   * @var \Drupal\recurly\RecurlyClientFactory
   */
  protected $mockClientFactory;

  /**
   * Mock subscription data.
   *
   * @var object
   */
  protected $subscription;

  /**
   * Add a mock recurly client factory to the container.
   *
   * Sets a mock recurly client factory that will return the client passed in
   * instead of the default one.
   *
   * @param \Recurly\Client $mock_client
   *   The mock client to use.
   */
  protected function setupMockRecurlyClient(Client $mock_client) {
    $this->mockClient = $mock_client;
    $this->mockClientFactory = $this->recurlyProphesize(RecurlyClientFactory::class);
    $this->mockClientFactory->getClientFromSettings()
      ->willReturn($this->mockClient);

    // Replace the recurly.client service with our mock.
    $this->container->set('recurly.client', $this->mockClientFactory->reveal());
    \Drupal::getContainer()->set('recurly.client', $this->mockClientFactory->reveal());
  }

  /**
   * Create a mock Recurly account resource.
   *
   * @param array $data
   *   An array of key-value pairs to set on the mock account. Any keys not
   *   present in the array will be filled in with default values.
   *
   * @return \Recurly\Resources\Account
   *   The mock account resource.
   */
  protected function getMockAccount(array $data = []) {
    $default_data = [
      'object' => 'account',
      'code' => '123-asd',
      'id' => '123-asd',
      'state' => 'active',
      'email' => '123-asd@example.com',
    ];
    $data = array_merge($default_data, $data);
    return $this->createMockResourceFromArray($data);
  }

  /**
   * Create a mock response for a Recurly API call.
   *
   * Creates a Recurly\Response object with the given data and headers.
   *
   * If no headers are given, the response will be a 200 OK response.
   *
   * @param string $data
   *   The JSON formatted data to include as the content of the response.
   * @param array $headers
   *   Optional headers to include in the response. Defaults to 200 OK.
   *
   * @return \Recurly\Response
   *   The mock response.
   */
  protected function createMockRecurlyResponse(string $data, array $headers = []) {
    // We need to mock a Request so that  we can use RecurlyResource to generate
    // a resource object. It doesn't matter what's in the request since we're
    // not actually issuing it.
    $options = [
      'params' => [
        'param-1' => 1,
        'param-2' => 'Param 2',
      ],
      'headers' => [
        'header-1' => 'Header 1',
        'header-2' => 'Header 2',
        'core-header-1' => 'Should be overridden',
      ],
      'core_headers' => [
        'core-header-1' => 'Core Header 1',
        'core-header-2' => 'Core Header 2',
      ],
    ];
    $body = [
      'code' => 'account-code',
      'first_name' => 'Wesley',
      'last_name' => 'Elizabeth',
      'email' => 'elbow@example.com',
    ];
    $request = new Request(
      'GET',
      '/accounts',
      $body,
      $options
    );

    $response = new Response($data, $request);

    $default_headers = ['HTTP/1.1 200 OK'];
    $headers = array_merge($headers, $default_headers);

    $response->setHeaders($headers);
    return $response;
  }

  /**
   * Creates a Recurly resource from a JSON string.
   *
   * @param string $resource
   *   A JSON string representing the Recurly resource.
   * @param array $headers
   *   An array of headers to include in the response.
   *
   * @return \Recurly\RecurlyResource
   *   The mock Recurly resource.
   */
  protected function createMockResourceFromJson(string $resource, array $headers = []): RecurlyResource {
    $response = RecurlyV3MockClient::createMockRecurlyResponse($resource, $headers);
    return RecurlyResource::fromResponse($response);
  }

  /**
   * Creates a Recurly resource from an associative array.
   *
   * @param array $array
   *   An associative array representing the Recurly resource.
   *
   * @return \Recurly\RecurlyResource
   *   The mock Recurly resource.
   */
  protected function createMockResourceFromArray(array $array): RecurlyResource {
    return $this->createMockResourceFromJson(json_encode($array));
  }

  /**
   * Creates a mock Recurly pager object.
   *
   * @param array $data
   *   An associative array of data to be used in the pager. The array
   *   should contain a 'data' key with an array of objects.
   * @param string $path
   *   The path to use for the pager's requests. In this case just used to make
   *   the pager unique.
   *
   * @return \Recurly\Pager
   *   The mock pager object.
   */
  protected function createRecurlyPagedResponse($data, $path): Pager {
    $count = count($data['data']);
    $client = $this->recurlyProphesize(Client::class);

    $count_response = RecurlyV3MockClient::createMockRecurlyResponse(
      json_encode($data),
      ['HTTP/1.1 200 OK', 'Recurly-Total-Records: ' . $count]
    );
    $client->pagerCount(Argument::containingString($path), Argument::cetera())
      ->willReturn($count_response);

    $client->nextPage(Argument::containingString($path), Argument::cetera())
      ->willReturn($count_response->toResource());

    $pager = new Pager($client->reveal(), $path);

    return $pager;
  }

  /**
   * Rebuilds entity definitions and related routes after config changes.
   */
  protected function rebuildRecurlyEntityRouting(): void {
    \Drupal::entityTypeManager()->clearCachedDefinitions();
    if (\Drupal::hasService('router.builder')) {
      \Drupal::service('router.builder')->rebuild();
    }
  }

  /**
   * Create a Drupal user account with an active Recurly subscription.
   *
   * Wrapper for \Drupal\Tests\user\Traits\UserCreationTrait::createUser() that
   * will also associate a recurly subscription with the user.
   *
   * Populate the Recurly Mock API client with the appropriate responses for
   * the account and its subscription.
   */
  public function createUserWithSubscription($permissions = []) {
    $permissions = empty($permissions) ? ['manage recurly subscription'] : $permissions;
    $account = $this->drupalCreateUser($permissions);

    // Add a Recurly subscription to the user.
    $recurly_account_code = 'user-' . $account->id();
    recurly_account_save_local(['code' => $recurly_account_code], 'user', (int) $account->id());

    // Populate the API client with a response for this account.
    $recurly_account = json_decode(RecurlyV3MockClient::loadRecurlyJsonFixture('accounts/show-200'));
    $recurly_account->id = $recurly_account_code;
    $recurly_account->code = $recurly_account_code;
    RecurlyV3MockClient::addResponse('GET', '/accounts/' . $recurly_account_code, json_encode($recurly_account));
    // Allow the account to be retrieved by ID or code. The Drupal module does
    // both in different places.
    RecurlyV3MockClient::addResponse('GET', '/accounts/code-' . $recurly_account_code, json_encode($recurly_account));

    // Add a subscription that can be returned for both getSubscription() and
    // listAccountSubscriptions() for the account.
    $recurly_subscription = json_decode(RecurlyV3MockClient::loadRecurlyJsonFixture('subscriptions/show-200'));
    $recurly_subscription->account_code = $recurly_account_code;
    $recurly_subscription->current_period_started_at = date('c', strtotime('-2 weeks'));
    $recurly_subscription->current_period_ends_at = date('c', strtotime('+2 weeks'));
    RecurlyV3MockClient::addResponse('GET', '/subscriptions/' . $recurly_subscription->id, json_encode($recurly_subscription));
    RecurlyV3MockClient::addResponse('GET', '/subscriptions/uuid-' . $recurly_subscription->uuid, json_encode($recurly_subscription));
    $subscription_list = [
      'object' => 'list',
      'has_more' => FALSE,
      'next' => NULL,
      'data' => [$recurly_subscription],
    ];
    RecurlyV3MockClient::addResponse('GET', '/accounts/' . $recurly_account_code . '/subscriptions', json_encode($subscription_list));
    RecurlyV3MockClient::addResponse('GET', '/accounts/code-' . $recurly_account_code . '/subscriptions', json_encode($subscription_list));

    $this->subscription = $recurly_subscription;

    return $account;
  }

}
