# Recurly Commerce API

Provides integration with Recurly Commerce API for subscription billing.

**Important**: This module is for **Recurly Commerce** (Shopify integration), not standard Recurly. They have completely different APIs and authentication methods.

## Overview

Recurly Commerce is Recurly's Shopify-specific subscription platform. This module provides:

- API client for Recurly Commerce REST API
- Webhook handling with signature verification
- Drush commands for API testing and webhook management
- Event dispatching system for webhook events

## Documentation

- **Recurly Commerce Docs**: https://docs.recurly.com/recurly-commerce/docs
- **API Reference**: https://docs.recurly.com/recurly-commerce/reference
- **Webhooks**: https://docs.recurly.com/recurly-commerce/docs/webhooks
- **Customer Portal**: https://docs.recurly.com/recurly-commerce/docs/customer-portal

## Requirements

- Drupal 9.4+ / 10 / 11
- Key module (for secure credential storage)

## Installation

1. Enable the module:
   ```bash
   drush en recurly_commerce_api
   ```

2. Configure API credentials at:
   `/admin/config/services/recurly-commerce-api`

3. Create API key in Recurly Commerce admin at https://admin.tryprive.com

4. Store the API key using the Key module

## Configuration

### API Settings

- **API Key**: Bearer token from Recurly Commerce admin (required)
- **Webhook Signing Key**: Used to verify incoming webhooks (optional, provided when creating webhooks)
- **Log Webhooks**: Enable detailed logging of webhook events

## Drush Commands

### Test API Connection

```bash
drush recurly:test-connection
```

Tests the API connection and displays basic information.

### List Subscriptions

```bash
drush recurly:list-subscriptions
drush recurly:list-subscriptions --limit=20
drush recurly:list-subscriptions --status=active
```

Lists subscriptions from Recurly Commerce.

### Webhook Management

**Important**: Recurly Commerce only supports **ONE topic per webhook**. You need to create separate webhooks for each event type.

#### Create Webhooks

```bash
# Create separate webhooks for each event type
drush recurly:create-webhook https://example.com/recurly/webhook --events="subscriptions/created"
drush recurly:create-webhook https://example.com/recurly/webhook --events="orders/created"
drush recurly:create-webhook https://example.com/recurly/webhook --events="billingAttempts/success"
drush recurly:create-webhook https://example.com/recurly/webhook --events="billingAttempts/failed"
```

#### Available Webhook Topics

- `subscriptions/created` - New subscription created
- `subscriptions/line_removed` - Line item removed from subscription
- `subscriptions/status/updated` - Subscription status changed
- `orders/created` - New order created
- `billingAttempts/created` - Billing attempt initiated
- `billingAttempts/success` - Successful billing
- `billingAttempts/failed` - Failed billing attempt
- `paymentMethod/updated` - Payment method updated
- `subscriptionActivities/created` - Subscription activity logged
- `subscription/customAttributes/updated` - Custom attributes modified

#### List Webhooks

```bash
drush recurly:list-webhooks
```

#### Delete Webhook

```bash
drush recurly:delete-webhook WEBHOOK_ID
```

## Webhook Handling

### Webhook Endpoint

Webhooks are received at: `/recurly/webhook`

### Signature Verification

All incoming webhooks are verified using HMAC-SHA256 signature in the `X-Recurly-Signature` header.

Format: `timestamp.signature`

### Event Dispatching

When a valid webhook is received, the module dispatches a Symfony event that can be subscribed to:

```php
use Drupal\recurly_commerce_api\Event\RecurlyApiWebhookEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class MyWebhookSubscriber implements EventSubscriberInterface {

  public static function getSubscribedEvents() {
    return [
      'recurly_api.webhook' => 'onWebhook',
    ];
  }

  public function onWebhook(RecurlyApiWebhookEvent $event) {
    $event_type = $event->getEventType();
    $data = $event->getData();

    // Handle webhook...
  }
}
```

## API Usage

### In Custom Code

```php
// Get the service
$recurly_api = \Drupal::service('recurly_commerce_api');

// Make API calls
$subscriptions = $recurly_api->get('/subscriptions', ['limit' => 10]);
$webhook = $recurly_api->post('/webhooks', [
  'address' => 'https://example.com/webhook',
  'topic' => 'subscriptions/created',
]);
$result = $recurly_api->patch('/subscriptions/123/status', ['status' => 'paused']);
$recurly_api->delete('/webhooks/abc123');
```

## Architecture

### Service: `recurly_commerce_api`

Main API client using Guzzle HTTP client with Bearer token authentication.

**Base URL**: `https://subs.api.tryprive.com/api/v1`

### Controller: `RecurlyApiWebhook`

Handles incoming webhook POST requests, validates signatures, and dispatches events.

### Drush Commands: `RecurlyApiCommands`

Command-line utilities for API testing and webhook management.

## Differences from Standard Recurly

| Feature | Standard Recurly | Recurly Commerce |
|---------|------------------|------------------|
| API Base URL | `v3.recurly.com` | `subs.api.tryprive.com/api/v1` |
| Authentication | HTTP Basic Auth | Bearer Token |
| Client Library | `recurly/recurly-client` | REST API (Guzzle) |
| Admin Portal | recurly.com | admin.tryprive.com |
| Webhook Topics | Multiple per webhook | **One per webhook** |
| Platform | Universal | Shopify-specific |

## Testing

Automated tests are available:

```bash
# Run all tests for this module
vendor/bin/phpunit web/modules/contrib/recurly_commerce_api/tests/
```

Tests include:
- API service with mocked HTTP responses
- Webhook signature validation
- Error handling

## Troubleshooting

### API Key Invalid

- Verify the API key is from Recurly Commerce (admin.tryprive.com)
- Standard Recurly API keys will NOT work
- Ensure the key is configured via the Key module

### Webhook Signature Validation Fails

- Verify the webhook signing key is correctly configured
- Check that the webhook was created via the Recurly Commerce API
- Ensure the `X-Recurly-Signature` header is present

### Connection Errors

- Check firewall/network access to `subs.api.tryprive.com`
- Verify SSL certificates are up to date
- Check Drupal logs: `drush watchdog:show --type=recurly_commerce_api`

## Support

For Recurly Commerce API issues, refer to:
- https://docs.recurly.com/recurly-commerce/docs
- https://docs.recurly.com/recurly-commerce/reference

## License

GPL-2.0-or-later
