<?php

namespace Drupal\recurly_commerce_api\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\recurly_commerce_api\Event\RecurlyCommerceApiWebhookEvent;
use Drupal\recurly_commerce_api\RecurlyCommerceApi;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Recurly API Webhook controller.
 *
 * Provides the route functionality for recurly_api.webhook route.
 */
class RecurlyCommerceApiWebhook extends ControllerBase {

  /**
   * Recurly API service.
   *
   * @var \Drupal\recurly_commerce_api\RecurlyCommerceApi
   */
  protected $recurlyApi;

  /**
   * {@inheritdoc}
   */
  public function __construct(RecurlyCommerceApi $recurly_api) {
    $this->recurlyApi = $recurly_api;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('recurly_commerce_api')
    );
  }

  /**
   * Captures the incoming webhook request.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   A Response object.
   */
  public function handleIncomingWebhook(Request $request) {
    // Handle HEAD/GET requests from Recurly to verify endpoint.
    // Symfony converts HEAD to GET automatically.
    if (in_array($request->getMethod(), ['HEAD', 'GET'])) {
      return new Response('OK', Response::HTTP_OK);
    }

    $input = $request->getContent();
    $decoded_input = json_decode($input);

    if (!$this->isValidWebhook($request)) {
      $this->getLogger('recurly_commerce_api')
        ->error('Invalid webhook signature. Is your webhook signing key set correctly? Event data: @data', [
          '@data' => $input,
        ]);
      return new Response(
        json_encode(['message' => 'Invalid signature']),
        Response::HTTP_FORBIDDEN
      );
    }

    $event = $decoded_input;
    $config = $this->config('recurly_commerce_api.settings');

    if ($config->get('log_webhooks')) {
      /** @var \Drupal\Core\Logger\LoggerChannelInterface $logger */
      $logger = $this->getLogger('recurly_commerce_api');
      $logger->info("Recurly webhook received event:\n @event", ['@event' => print_r($event, TRUE)]);
    }

    // Dispatch the webhook event.
    try {
      $dispatcher = \Drupal::service('event_dispatcher');
      // Recurly Commerce uses 'topic' instead of 'event_type'
      $event_type = $decoded_input->topic ?? $decoded_input->event_type ?? 'unknown';
      $webhook_event = new RecurlyCommerceApiWebhookEvent($event_type, $decoded_input);
      $dispatcher->dispatch($webhook_event, 'recurly_commerce_api.webhook');

      return new Response('OK', Response::HTTP_OK);
    }
    catch (\Exception $e) {
      // Log the error and return 500 so Recurly knows to retry
      $this->getLogger('recurly_commerce_api')
        ->error('Webhook processing failed: @error', [
          '@error' => $e->getMessage(),
        ]);

      // Return 500 Internal Server Error to trigger Recurly retry
      return new Response(
        'Webhook processing failed: ' . $e->getMessage(),
        Response::HTTP_INTERNAL_SERVER_ERROR
      );
    }
  }

  /**
   * Determines if a webhook is valid.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return bool
   *   TRUE if valid, FALSE otherwise.
   */
  private function isValidWebhook(Request $request): bool {
    $config = $this->config('recurly_commerce_api.settings');

    // Skip validation if configured to do so.
    if ($config->get('skip_webhook_validation')) {
      $this->getLogger('recurly_commerce_api')->warning('Webhook signature validation is disabled. This should only be used for development/testing.');
      return TRUE;
    }

    // Check for signature header (Recurly Commerce uses X-Prive-Hmac-Sha256).
    if (!$request->headers->has('X-Prive-Hmac-Sha256')) {
      $this->getLogger('recurly_commerce_api')->warning('Webhook received without X-Prive-Hmac-Sha256 header.');
      return FALSE;
    }

    return $this->isWebhookSignatureValid($request);
  }

  /**
   * Validates the webhook signature.
   *
   * Recurly Commerce signs webhooks using HMAC-SHA256. The signature is sent
   * in the X-Prive-Hmac-Sha256 header as a base64-encoded hash of the body.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return bool
   *   TRUE if signature is valid, FALSE otherwise.
   *
   * @see https://docs.recurly.com/recurly-commerce/docs/webhooks
   */
  private function isWebhookSignatureValid(Request $request): bool {
    $signing_key = $this->recurlyApi->getWebhookSigningKey();
    if (!$signing_key) {
      $this->getLogger('recurly_commerce_api')->warning('Webhook signing key not configured.');
      return FALSE;
    }

    $received_signature = $request->headers->get('X-Prive-Hmac-Sha256');

    // Calculate expected signature: HMAC-SHA256 of the request body, base64-encoded
    $body = $request->getContent();
    $expected_signature = base64_encode(hash_hmac('sha256', $body, $signing_key, TRUE));

    // Use timing-safe comparison
    $valid = hash_equals($expected_signature, $received_signature);

    if (!$valid) {
      $this->getLogger('recurly_commerce_api')->error('Webhook signature validation failed. Expected: @expected, Received: @received', [
        '@expected' => $expected_signature,
        '@received' => $received_signature,
      ]);
    }

    return $valid;
  }

}
