<?php

namespace Drupal\recurly_commerce_api\Drush\Commands;

use Consolidation\OutputFormatters\StructuredData\RowsOfFields;
use Drupal\recurly_commerce_api\RecurlyCommerceApi;
use Drush\Commands\DrushCommands;

/**
 * Recurly Commerce API Drush commands.
 */
class RecurlyCommerceApiCommands extends DrushCommands {

  /**
   * The Recurly Commerce API service.
   *
   * @var \Drupal\recurly_commerce_api\RecurlyCommerceApi
   */
  private RecurlyCommerceApi $recurlyApi;

  /**
   * RecurlyCommerceApiCommands constructor.
   *
   * @param \Drupal\recurly_commerce_api\RecurlyCommerceApi $recurly_api
   *   The Recurly Commerce API service.
   */
  public function __construct(RecurlyCommerceApi $recurly_api) {
    parent::__construct();
    $this->recurlyApi = $recurly_api;
  }

  /**
   * Test the Recurly Commerce API connection.
   *
   * @command recurly:test-connection
   *
   * @usage recurly:test-connection
   *   Test the Recurly Commerce API connection by fetching subscriptions.
   *
   * @aliases rtest
   */
  public function testConnection() {
    $this->logger()->info('Testing Recurly Commerce API connection...');

    try {
      // Test the connection by fetching one subscription
      $result = $this->recurlyApi->get('/subscriptions', ['limit' => 1]);

      if ($result === NULL) {
        $this->output()->writeln("<error>Connection failed. Check the error log for details.</error>");
        return DrushCommands::EXIT_FAILURE;
      }

      $this->output()->writeln("<info>Successfully connected to Recurly Commerce API!</info>");
      $this->output()->writeln("<info>Base URL: https://subs.api.tryprive.com/api/v1</info>");
      $this->output()->writeln("<info>API Key: " . substr($this->recurlyApi->getApiKey(), 0, 10) . "...</info>");

      if (isset($result['data']) && count($result['data']) > 0) {
        $subscription = $result['data'][0];
        $this->output()->writeln("<comment>Sample subscription: {$subscription['id']}</comment>");
      }
      else {
        $this->output()->writeln("<comment>No subscriptions found.</comment>");
      }

    }
    catch (\Exception $e) {
      $this->logger()->error('Connection test failed: @error', ['@error' => $e->getMessage()]);
      $this->output()->writeln("<error>Error: Could not connect to Recurly Commerce API.</error>");
      $this->output()->writeln("<error>Message: " . $e->getMessage() . "</error>");
      return DrushCommands::EXIT_FAILURE;
    }
  }

  /**
   * List Recurly Commerce subscriptions.
   *
   * @command recurly:list-subscriptions
   *
   * @param array $options
   *   Command options.
   *
   * @option limit
   *   Number of subscriptions to return (default: 10).
   * @option status
   *   Filter by status (active, canceled, etc.).
   *
   * @field-labels
   *   id: ID
   *   status: Status
   *   created_at: Created At
   * @default-fields id,status,created_at
   *
   * @usage recurly:list-subscriptions
   *   List up to 10 subscriptions.
   * @usage recurly:list-subscriptions --limit=20
   *   List up to 20 subscriptions.
   *
   * @aliases rlists
   *
   * @return \Consolidation\OutputFormatters\StructuredData\RowsOfFields
   */
  public function listSubscriptions(array $options = ['limit' => 10, 'status' => NULL]): RowsOfFields {
    $this->logger()->info('Fetching Recurly Commerce subscriptions...');

    try {
      $params = ['limit' => $options['limit']];
      if ($options['status']) {
        $params['status'] = $options['status'];
      }

      $result = $this->recurlyApi->get('/subscriptions', $params);

      if ($result === NULL) {
        $this->output()->writeln("<error>Failed to fetch subscriptions.</error>");
        return new RowsOfFields([]);
      }

      $rows = [];
      if (isset($result['data'])) {
        foreach ($result['data'] as $subscription) {
          $rows[] = [
            'id' => $subscription['id'] ?? '',
            'status' => $subscription['status'] ?? '',
            'created_at' => $subscription['createdAt'] ?? '',
          ];
        }
      }

      if (empty($rows)) {
        $this->output()->writeln('<comment>No subscriptions found.</comment>');
      }

      return new RowsOfFields($rows);

    }
    catch (\Exception $e) {
      $this->logger()->error('Failed to list subscriptions: @error', ['@error' => $e->getMessage()]);
      $this->output()->writeln("<error>Error: " . $e->getMessage() . "</error>");
      return new RowsOfFields([]);
    }
  }

  /**
   * List configured webhooks.
   *
   * @command recurly:list-webhooks
   *
   * @field-labels
   *   id: ID
   *   address: Address
   *   topic: Topic
   *   status: Status
   * @default-fields id,address,topic,status
   *
   * @usage recurly:list-webhooks
   *   List all configured webhooks.
   *
   * @aliases rlistw
   *
   * @return \Consolidation\OutputFormatters\StructuredData\RowsOfFields
   */
  public function listWebhooks(): RowsOfFields {
    $this->logger()->info('Fetching webhooks...');

    try {
      $result = $this->recurlyApi->get('/webhooks');

      if ($result === NULL) {
        $this->output()->writeln("<error>Failed to fetch webhooks.</error>");
        return new RowsOfFields([]);
      }

      $rows = [];
      if (isset($result['webhooks'])) {
        foreach ($result['webhooks'] as $webhook) {
          $rows[] = [
            'id' => $webhook['id'] ?? '',
            'address' => $webhook['address'] ?? '',
            'topic' => $webhook['topic'] ?? '',
            'status' => $webhook['status'] ?? '',
          ];
        }
      }

      if (empty($rows)) {
        $this->output()->writeln('<comment>No webhooks found.</comment>');
      }

      return new RowsOfFields($rows);

    }
    catch (\Exception $e) {
      $this->logger()->error('Failed to list webhooks: @error', ['@error' => $e->getMessage()]);
      $this->output()->writeln("<error>Error: " . $e->getMessage() . "</error>");
      return new RowsOfFields([]);
    }
  }

  /**
   * Create a webhook.
   *
   * Note: Recurly Commerce only supports ONE topic per webhook.
   * If multiple events are provided, only the first will be used.
   *
   * @command recurly:create-webhook
   *
   * @param string $url
   *   The webhook URL.
   * @param array $options
   *   Command options.
   *
   * @option events
   *   Event topic (default: subscriptions/created). Available: subscriptions/created, subscriptions/line_removed, subscriptions/status/updated, orders/created, billingAttempts/created, billingAttempts/success, billingAttempts/failed, paymentMethod/updated, subscriptionActivities/created, subscription/customAttributes/updated.
   *
   * @usage recurly:create-webhook https://example.com/webhook
   *   Create a webhook for subscriptions/created event.
   * @usage recurly:create-webhook https://example.com/webhook --events="subscriptions/status/updated"
   *   Create a webhook for subscription status updates.
   *
   * @aliases rcreatew
   */
  public function createWebhook(string $url, array $options = ['events' => 'subscriptions/created,orders/created,billingAttempts/success,billingAttempts/failed']) {
    $this->logger()->info('Creating webhook...');

    try {
      $events = explode(',', $options['events']);
      $events = array_map('trim', $events);

      // Recurly Commerce API only supports one topic per webhook
      $topic = $events[0];

      $data = [
        'address' => $url,
        'topic' => $topic,
      ];

      $result = $this->recurlyApi->post('/webhooks', $data);

      if ($result === NULL) {
        $this->output()->writeln("<error>Failed to create webhook.</error>");
        return DrushCommands::EXIT_FAILURE;
      }

      $this->output()->writeln("<info>Webhook created successfully!</info>");
      if (isset($result['id'])) {
        $this->output()->writeln("<comment>Webhook ID: {$result['id']}</comment>");
      }
      if (isset($result['topic'])) {
        $this->output()->writeln("<comment>Topic: {$result['topic']}</comment>");
      }
      if (isset($result['address'])) {
        $this->output()->writeln("<comment>Address: {$result['address']}</comment>");
      }
      $this->output()->writeln("");
      $this->output()->writeln("<info>Note: Webhook signing key must be obtained from Recurly Commerce admin panel.</info>");
      $this->output()->writeln("<info>Configure it at /admin/config/services/recurly-commerce-api</info>");

    }
    catch (\Exception $e) {
      $this->logger()->error('Failed to create webhook: @error', ['@error' => $e->getMessage()]);
      $this->output()->writeln("<error>Error: " . $e->getMessage() . "</error>");
      return DrushCommands::EXIT_FAILURE;
    }
  }

  /**
   * Delete a webhook.
   *
   * @command recurly:delete-webhook
   *
   * @param string $webhook_id
   *   The webhook ID to delete.
   *
   * @usage recurly:delete-webhook abc123
   *   Delete webhook with ID abc123.
   *
   * @aliases rdeletew
   */
  public function deleteWebhook(string $webhook_id) {
    $this->logger()->info('Deleting webhook...');

    try {
      $result = $this->recurlyApi->delete("/webhooks/{$webhook_id}");

      if ($result === NULL) {
        $this->output()->writeln("<error>Failed to delete webhook.</error>");
        return DrushCommands::EXIT_FAILURE;
      }

      $this->output()->writeln("<info>Webhook deleted successfully!</info>");

    }
    catch (\Exception $e) {
      $this->logger()->error('Failed to delete webhook: @error', ['@error' => $e->getMessage()]);
      $this->output()->writeln("<error>Error: " . $e->getMessage() . "</error>");
      return DrushCommands::EXIT_FAILURE;
    }
  }

}
