<?php

namespace Drupal\recurly_commerce_api\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\recurly_commerce_api\RecurlyCommerceApi;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Recurly API admin form.
 *
 * Contains admin form functionality for the Recurly API.
 */
class RecurlyCommerceApiAdminForm extends ConfigFormBase {

  /**
   * Recurly API service.
   *
   * @var \Drupal\recurly_commerce_api\RecurlyCommerceApi
   */
  protected $recurlyApi;

  /**
   * Constructs a RecurlyCommerceApiAdminForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager.
   * @param \Drupal\recurly_commerce_api\RecurlyCommerceApi $recurly_api
   *   Recurly API service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, RecurlyCommerceApi $recurly_api) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->recurlyApi = $recurly_api;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('recurly_commerce_api')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'recurly_api_admin_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'recurly_commerce_api.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('recurly_commerce_api.settings');

    $form['info'] = [
      '#type' => 'markup',
      '#markup' => $this->t('<p><strong>Recurly Commerce (Shopify Integration)</strong></p><p>This module connects to Recurly Commerce API at <code>https://subs.api.tryprive.com/api/v1</code></p>'),
    ];

    $form['api_key'] = [
      '#type' => 'key_select',
      '#title' => $this->t('Recurly Commerce API Key (Bearer Token)'),
      '#description' => $this->t('API key from Recurly Commerce admin at <a href="https://admin.tryprive.com" target="_blank">admin.tryprive.com</a>. This is a Bearer token used for API authentication.'),
      '#default_value' => $config->get('api_key'),
      '#required' => TRUE,
    ];

    $form['webhook_signing_key'] = [
      '#type' => 'key_select',
      '#title' => $this->t('Webhook Signing Key'),
      '#description' => $this->t('Used to verify incoming webhooks (different from API key). In Recurly Commerce, you can create webhooks via the API (see drush commands) which will provide the signing key.'),
      '#default_value' => $config->get('webhook_signing_key'),
    ];

    $form['skip_webhook_validation'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Skip webhook signature validation'),
      '#description' => $this->t('<strong>Warning:</strong> Only use this for development/testing. Disabling signature validation allows any external service to send webhook requests to your site.'),
      '#default_value' => $config->get('skip_webhook_validation'),
    ];

    $form['webhook_url'] = [
      '#type' => 'textfield',
      '#disabled' => TRUE,
      '#title' => $this->t('Webhook URL'),
      '#default_value' => Url::fromRoute('recurly_commerce_api.webhook', [], ['absolute' => TRUE])->toString(),
      '#description' => $this->t('Use drush commands to create webhooks via API (e.g., <code>drush recurly:create-webhook</code>). Available events: subscriptions/created, orders/created, billingAttempts/success, billingAttempts/failed, etc.'),
    ];

    $form['log_webhooks'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log incoming webhooks'),
      '#description' => $this->t('Enable detailed logging of webhook events. Useful for debugging but may create large log files.'),
      '#default_value' => $config->get('log_webhooks'),
    ];

    $form['portal_base_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Customer Portal Base URL'),
      '#description' => $this->t('Base URL for your Shopify store (e.g., <code>https://store.guitargate.com</code> or <code>https://582300c.myshopify.com</code>). The portal path (<code>/apps/customers/portal</code>) and token will be appended automatically.'),
      '#default_value' => $config->get('portal_base_url') ?: '',
      '#required' => TRUE,
    ];

    if ($this->recurlyApi->getApiKey()) {
      $form['recurly_test'] = [
        '#type' => 'button',
        '#value' => $this->t('Test Recurly Connection'),
        '#ajax' => [
          'callback' => [$this, 'testRecurlyConnection'],
          'wrapper' => 'recurly-connect-results',
          'method' => 'append',
        ],
        '#suffix' => '<div id="recurly-connect-results"></div>',
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('recurly_commerce_api.settings')
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('webhook_signing_key', $form_state->getValue('webhook_signing_key'))
      ->set('skip_webhook_validation', $form_state->getValue('skip_webhook_validation'))
      ->set('log_webhooks', $form_state->getValue('log_webhooks'))
      ->set('portal_base_url', $form_state->getValue('portal_base_url'))
      ->save();
    parent::submitForm($form, $form_state);
  }

  /**
   * AJAX callback to test the Recurly connection.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   The form structure.
   */
  public function testRecurlyConnection(array &$form, FormStateInterface $form_state): array {
    try {
      // Simple API call to verify connection - list subscriptions with limit 1
      $result = $this->recurlyApi->get('/subscriptions', ['limit' => 1]);

      if ($result !== NULL) {
        $count = isset($result['data']) ? count($result['data']) : 0;
        return ['#markup' => $this->t('<div class="messages messages--status">Success! Connected to Recurly Commerce API. Found @count subscriptions.</div>', ['@count' => $count])];
      }
      else {
        return ['#markup' => $this->t('<div class="messages messages--error">Error! Could not connect to Recurly Commerce API. Check the error log for details.</div>')];
      }
    }
    catch (\Exception $e) {
      \Drupal::logger('recurly_commerce_api')->error('Connection test failed: @error', ['@error' => $e->getMessage()]);
      return ['#markup' => $this->t('<div class="messages messages--error">Error! Could not connect to Recurly Commerce. Check the error log for details.</div>')];
    }
  }

}
