<?php

namespace Drupal\recurly_commerce_api;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\key\KeyRepositoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Psr\Log\LoggerInterface;

/**
 * Recurly Commerce API service.
 *
 * Note: This is for Recurly Commerce (Shopify integration), not standard Recurly.
 * Base URL: https://subs.api.tryprive.com/api/v1
 *
 * @package Drupal\recurly_commerce_api
 */
class RecurlyCommerceApi {

  /**
   * Recurly Commerce API base URL.
   */
  const BASE_URL = 'https://subs.api.tryprive.com/api/v1';

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * Logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The Key Repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected $key;

  /**
   * The configuration object.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  private $config;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   * @param \Drupal\key\KeyRepositoryInterface $key
   *   The key repository.
   */
  public function __construct(ConfigFactoryInterface $config_factory, ClientInterface $http_client, LoggerInterface $logger, KeyRepositoryInterface $key) {
    $this->config = $config_factory->get('recurly_commerce_api.settings');
    $this->httpClient = $http_client;
    $this->logger = $logger;
    $this->key = $key;
  }

  /**
   * Gets Recurly API key (Bearer token).
   *
   * @return string|null
   *   Recurly API key.
   */
  public function getApiKey(): ?string {
    $key_id = $this->config->get('api_key');
    if ($key_id) {
      $key_entity = $this->key->getKey($key_id);
      if ($key_entity) {
        return $key_entity->getKeyValue();
      }
    }

    return NULL;
  }

  /**
   * Gets Recurly webhook signing key.
   *
   * @return string|null
   *   Webhook signing key.
   */
  public function getWebhookSigningKey(): ?string {
    $key_id = $this->config->get('webhook_signing_key');
    if ($key_id) {
      $key_entity = $this->key->getKey($key_id);
      if ($key_entity) {
        return $key_entity->getKeyValue();
      }
    }

    return NULL;
  }

  /**
   * Makes an authenticated GET request to Recurly Commerce API.
   *
   * @param string $endpoint
   *   The API endpoint (e.g., '/subscriptions').
   * @param array $params
   *   Query parameters.
   *
   * @return array|null
   *   The response data or NULL on failure.
   */
  public function get(string $endpoint, array $params = []): ?array {
    return $this->request('GET', $endpoint, ['query' => $params]);
  }

  /**
   * Makes an authenticated POST request to Recurly Commerce API.
   *
   * @param string $endpoint
   *   The API endpoint (e.g., '/webhooks').
   * @param array $data
   *   The request body data.
   *
   * @return array|null
   *   The response data or NULL on failure.
   */
  public function post(string $endpoint, array $data = []): ?array {
    return $this->request('POST', $endpoint, ['json' => $data]);
  }

  /**
   * Makes an authenticated PATCH request to Recurly Commerce API.
   *
   * @param string $endpoint
   *   The API endpoint (e.g., '/subscriptions/123/status').
   * @param array $data
   *   The request body data.
   *
   * @return array|null
   *   The response data or NULL on failure.
   */
  public function patch(string $endpoint, array $data = []): ?array {
    return $this->request('PATCH', $endpoint, ['json' => $data]);
  }

  /**
   * Makes an authenticated DELETE request to Recurly Commerce API.
   *
   * @param string $endpoint
   *   The API endpoint (e.g., '/webhooks/123').
   *
   * @return array|null
   *   The response data or NULL on failure.
   */
  public function delete(string $endpoint): ?array {
    return $this->request('DELETE', $endpoint);
  }

  /**
   * Makes an authenticated HTTP request to Recurly Commerce API.
   *
   * @param string $method
   *   HTTP method (GET, POST, PATCH, DELETE).
   * @param string $endpoint
   *   The API endpoint.
   * @param array $options
   *   Request options (query params, json body, etc.).
   *
   * @return array|null
   *   The response data or NULL on failure.
   */
  private function request(string $method, string $endpoint, array $options = []): ?array {
    $api_key = $this->getApiKey();
    if (!$api_key) {
      $this->logger->error('Recurly API key not configured');
      return NULL;
    }

    $url = self::BASE_URL . $endpoint;

    // Add Bearer token authentication
    $options['headers'] = array_merge($options['headers'] ?? [], [
      'Authorization' => 'Bearer ' . $api_key,
      'Content-Type' => 'application/json',
      'Accept' => 'application/json',
    ]);

    try {
      $response = $this->httpClient->request($method, $url, $options);
      $body = (string) $response->getBody();

      if (empty($body)) {
        return [];
      }

      return json_decode($body, TRUE);
    }
    catch (RequestException $e) {
      $this->logger->error('Recurly API request failed: @message', [
        '@message' => $e->getMessage(),
      ]);

      if ($e->hasResponse()) {
        $error_body = (string) $e->getResponse()->getBody();
        $this->logger->error('Response: @response', ['@response' => $error_body]);
      }

      return NULL;
    }
  }

}
