<?php

namespace Drupal\Tests\recurly_commerce_api\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\recurly_commerce_api\RecurlyCommerceApi;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;

/**
 * Tests for RecurlyCommerceApi service.
 *
 * @group recurly_commerce_api
 */
class RecurlyCommerceApiTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'key',
    'recurly_commerce_api',
  ];

  /**
   * The Recurly API service.
   *
   * @var \Drupal\recurly_commerce_api\RecurlyCommerceApi
   */
  protected $recurlyApi;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installConfig(['recurly_commerce_api']);

    // Mock the Key repository to return test values.
    $key_entity = $this->getMockBuilder('Drupal\key\Entity\Key')
      ->disableOriginalConstructor()
      ->getMock();
    $key_entity->method('getKeyValue')
      ->willReturn('test_api_key_12345');

    $key_repository = $this->getMockBuilder('Drupal\key\KeyRepositoryInterface')
      ->disableOriginalConstructor()
      ->getMock();
    $key_repository->method('getKey')
      ->willReturn($key_entity);

    $this->container->set('key.repository', $key_repository);

    // Configure the module to use a test key ID.
    $this->config('recurly_commerce_api.settings')
      ->set('api_key', 'test_recurly_key')
      ->save();

    // Create the RecurlyCommerceApi service with mocked dependencies.
    $config_factory = $this->container->get('config.factory');
    $http_client = $this->container->get('http_client');
    $logger = $this->container->get('logger.channel.recurly_commerce_api');

    $this->recurlyApi = new RecurlyCommerceApi($config_factory, $http_client, $logger, $key_repository);
  }

  /**
   * Tests successful API connection and subscription retrieval.
   */
  public function testSuccessfulApiConnection() {
    // Mock a successful API response.
    $mock_response = [
      'data' => [
        [
          'id' => 'sub_123',
          'status' => 'active',
          'customer' => [
            'email' => 'test@example.com',
          ],
        ],
      ],
      'meta' => [
        'total' => 1,
      ],
    ];

    $mock = new MockHandler([
      new Response(200, ['Content-Type' => 'application/json'], json_encode($mock_response)),
    ]);

    $handlerStack = HandlerStack::create($mock);
    $client = new Client(['handler' => $handlerStack]);

    // Create a new RecurlyCommerceApi service with mocked HTTP client.
    $config_factory = $this->container->get('config.factory');
    $logger = $this->container->get('logger.channel.recurly_commerce_api');
    $key_repository = $this->container->get('key.repository');

    $recurlyApi = new RecurlyCommerceApi($config_factory, $client, $logger, $key_repository);

    // Make API call.
    $result = $recurlyApi->get('/subscriptions', ['limit' => 1]);

    // Assert response structure.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('data', $result);
    $this->assertCount(1, $result['data']);
    $this->assertEquals('sub_123', $result['data'][0]['id']);
    $this->assertEquals('active', $result['data'][0]['status']);
  }

  /**
   * Tests API error handling for 401 Unauthorized.
   */
  public function testUnauthorizedApiResponse() {
    // Mock a 401 Unauthorized response.
    $mock = new MockHandler([
      new Response(401, ['Content-Type' => 'application/json'], json_encode([
        'error' => 'Unauthorized',
        'message' => 'Invalid API key',
      ])),
    ]);

    $handlerStack = HandlerStack::create($mock);
    $client = new Client(['handler' => $handlerStack]);

    // Create a new RecurlyCommerceApi service with mocked HTTP client.
    $config_factory = $this->container->get('config.factory');
    $logger = $this->container->get('logger.channel.recurly_commerce_api');
    $key_repository = $this->container->get('key.repository');

    $recurlyApi = new RecurlyCommerceApi($config_factory, $client, $logger, $key_repository);

    // Make API call - should return NULL on error.
    $result = $recurlyApi->get('/subscriptions');

    $this->assertNull($result);
  }

  /**
   * Tests API error handling for 500 Server Error.
   */
  public function testServerErrorApiResponse() {
    // Mock a 500 Server Error response.
    $mock = new MockHandler([
      new Response(500, ['Content-Type' => 'application/json'], json_encode([
        'error' => 'Internal Server Error',
      ])),
    ]);

    $handlerStack = HandlerStack::create($mock);
    $client = new Client(['handler' => $handlerStack]);

    // Create a new RecurlyCommerceApi service with mocked HTTP client.
    $config_factory = $this->container->get('config.factory');
    $logger = $this->container->get('logger.channel.recurly_commerce_api');
    $key_repository = $this->container->get('key.repository');

    $recurlyApi = new RecurlyCommerceApi($config_factory, $client, $logger, $key_repository);

    // Make API call - should return NULL on error.
    $result = $recurlyApi->get('/subscriptions');

    $this->assertNull($result);
  }

  /**
   * Tests POST request for webhook creation.
   */
  public function testWebhookCreation() {
    // Mock a successful webhook creation response.
    $mock_response = [
      'id' => 'wh_123456',
      'url' => 'https://example.com/webhook',
      'events' => ['subscriptions/created', 'orders/created'],
      'status' => 'active',
      'signingKey' => 'whsec_test_key_123',
    ];

    $mock = new MockHandler([
      new Response(201, ['Content-Type' => 'application/json'], json_encode($mock_response)),
    ]);

    $handlerStack = HandlerStack::create($mock);
    $client = new Client(['handler' => $handlerStack]);

    // Create a new RecurlyCommerceApi service with mocked HTTP client.
    $config_factory = $this->container->get('config.factory');
    $logger = $this->container->get('logger.channel.recurly_commerce_api');
    $key_repository = $this->container->get('key.repository');

    $recurlyApi = new RecurlyCommerceApi($config_factory, $client, $logger, $key_repository);

    // Make API call to create webhook.
    $result = $recurlyApi->post('/webhooks', [
      'url' => 'https://example.com/webhook',
      'events' => ['subscriptions/created'],
    ]);

    // Assert response structure.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('id', $result);
    $this->assertArrayHasKey('signingKey', $result);
    $this->assertEquals('wh_123456', $result['id']);
    $this->assertEquals('whsec_test_key_123', $result['signingKey']);
  }

  /**
   * Tests handling of missing API key.
   */
  public function testMissingApiKey() {
    // Mock a key repository that returns NULL for missing keys.
    $key_repository = $this->getMockBuilder('Drupal\key\KeyRepositoryInterface')
      ->disableOriginalConstructor()
      ->getMock();
    $key_repository->method('getKey')
      ->willReturn(NULL);

    $config_factory = $this->container->get('config.factory');
    $http_client = $this->container->get('http_client');
    $logger = $this->container->get('logger.channel.recurly_commerce_api');

    $recurlyApi = new RecurlyCommerceApi($config_factory, $http_client, $logger, $key_repository);

    // Make API call - should return NULL when API key is missing.
    $result = $recurlyApi->get('/subscriptions');

    $this->assertNull($result);
  }

}
