<?php

namespace Drupal\Tests\recurly_commerce_api\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\recurly_commerce_api\RecurlyCommerceApi;
use Symfony\Component\HttpFoundation\Request;

/**
 * Tests for webhook signature validation.
 *
 * @group recurly_commerce_api
 */
class WebhookValidationTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'key',
    'recurly_commerce_api',
  ];

  /**
   * The webhook controller.
   *
   * @var \Drupal\recurly_commerce_api\Controller\RecurlyCommerceApiWebhook
   */
  protected $webhookController;

  /**
   * Mock key repository.
   *
   * @var \PHPUnit\Framework\MockObject\MockObject
   */
  protected $keyRepository;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installConfig(['recurly_commerce_api']);

    // Mock the Key repository to return test webhook signing key.
    $key_entity = $this->getMockBuilder('Drupal\key\Entity\Key')
      ->disableOriginalConstructor()
      ->getMock();
    $key_entity->method('getKeyValue')
      ->willReturn('test_signing_key_12345');

    $this->keyRepository = $this->getMockBuilder('Drupal\key\KeyRepositoryInterface')
      ->disableOriginalConstructor()
      ->getMock();
    $this->keyRepository->method('getKey')
      ->willReturn($key_entity);

    $this->container->set('key.repository', $this->keyRepository);

    // Configure the module to use a test key ID.
    $this->config('recurly_commerce_api.settings')
      ->set('webhook_signing_key', 'test_webhook_key')
      ->save();
  }

  /**
   * Tests webhook validation with valid signature.
   */
  public function testValidWebhookSignature() {
    $payload = json_encode([
      'topic' => 'subscriptions/created',
      'data' => [
        'id' => 'sub_123',
        'status' => 'active',
      ],
    ]);

    // Calculate valid signature using HMAC-SHA256, base64 encoded.
    $signing_key = 'test_signing_key_12345';
    $signature = base64_encode(hash_hmac('sha256', $payload, $signing_key, TRUE));

    // Create request with valid signature header.
    $request = Request::create('/recurly/webhook', 'POST', [], [], [], [], $payload);
    $request->headers->set('X-Prive-Hmac-Sha256', $signature);
    $request->headers->set('Content-Type', 'application/json');

    // Create webhook controller instance.
    $config_factory = $this->container->get('config.factory');
    $http_client = $this->container->get('http_client');
    $logger = $this->container->get('logger.channel.recurly_commerce_api');

    $recurly_api = new RecurlyCommerceApi($config_factory, $http_client, $logger, $this->keyRepository);
    $controller = new \Drupal\recurly_commerce_api\Controller\RecurlyCommerceApiWebhook($recurly_api);

    // Handle the webhook.
    $response = $controller->handleIncomingWebhook($request);

    // Assert response is successful.
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertEquals('OK', $response->getContent());
  }

  /**
   * Tests webhook validation with invalid signature.
   */
  public function testInvalidWebhookSignature() {
    $payload = json_encode([
      'topic' => 'subscriptions/created',
      'data' => ['id' => 'sub_123'],
    ]);

    // Create request with INVALID signature.
    $request = Request::create('/recurly/webhook', 'POST', [], [], [], [], $payload);
    $request->headers->set('X-Prive-Hmac-Sha256', 'invalid_signature_base64_encoded');
    $request->headers->set('Content-Type', 'application/json');

    // Create webhook controller instance.
    $config_factory = $this->container->get('config.factory');
    $http_client = $this->container->get('http_client');
    $logger = $this->container->get('logger.channel.recurly_commerce_api');

    $recurly_api = new RecurlyCommerceApi($config_factory, $http_client, $logger, $this->keyRepository);
    $controller = new \Drupal\recurly_commerce_api\Controller\RecurlyCommerceApiWebhook($recurly_api);

    // Handle the webhook.
    $response = $controller->handleIncomingWebhook($request);

    // Assert response is forbidden.
    $this->assertEquals(403, $response->getStatusCode());
    $this->assertStringContainsString('Invalid signature', $response->getContent());
  }

  /**
   * Tests webhook validation without signature header.
   */
  public function testMissingSignatureHeader() {
    $payload = json_encode([
      'topic' => 'subscriptions/created',
      'data' => ['id' => 'sub_123'],
    ]);

    // Create request WITHOUT signature header.
    $request = Request::create('/recurly/webhook', 'POST', [], [], [], [], $payload);
    $request->headers->set('Content-Type', 'application/json');

    // Create webhook controller instance.
    $config_factory = $this->container->get('config.factory');
    $http_client = $this->container->get('http_client');
    $logger = $this->container->get('logger.channel.recurly_commerce_api');

    $recurly_api = new RecurlyCommerceApi($config_factory, $http_client, $logger, $this->keyRepository);
    $controller = new \Drupal\recurly_commerce_api\Controller\RecurlyCommerceApiWebhook($recurly_api);

    // Handle the webhook.
    $response = $controller->handleIncomingWebhook($request);

    // Assert response is forbidden.
    $this->assertEquals(403, $response->getStatusCode());
  }

  /**
   * Tests webhook validation with malformed signature format.
   */
  public function testMalformedSignatureFormat() {
    $payload = json_encode([
      'topic' => 'subscriptions/created',
      'data' => ['id' => 'sub_123'],
    ]);

    // Create request with malformed signature (not base64 encoded).
    $request = Request::create('/recurly/webhook', 'POST', [], [], [], [], $payload);
    $request->headers->set('X-Prive-Hmac-Sha256', 'not-a-valid-base64-signature!!!');
    $request->headers->set('Content-Type', 'application/json');

    // Create webhook controller instance.
    $config_factory = $this->container->get('config.factory');
    $http_client = $this->container->get('http_client');
    $logger = $this->container->get('logger.channel.recurly_commerce_api');

    $recurly_api = new RecurlyCommerceApi($config_factory, $http_client, $logger, $this->keyRepository);
    $controller = new \Drupal\recurly_commerce_api\Controller\RecurlyCommerceApiWebhook($recurly_api);

    // Handle the webhook.
    $response = $controller->handleIncomingWebhook($request);

    // Assert response is forbidden.
    $this->assertEquals(403, $response->getStatusCode());
  }

  /**
   * Tests webhook validation with missing signing key configuration.
   */
  public function testMissingSigningKey() {
    $payload = json_encode([
      'topic' => 'subscriptions/created',
      'data' => ['id' => 'sub_123'],
    ]);

    // Mock a key repository that returns NULL (no signing key configured).
    $key_repository = $this->getMockBuilder('Drupal\key\KeyRepositoryInterface')
      ->disableOriginalConstructor()
      ->getMock();
    $key_repository->method('getKey')
      ->willReturn(NULL);

    // Create request with signature.
    $request = Request::create('/recurly/webhook', 'POST', [], [], [], [], $payload);
    $request->headers->set('X-Prive-Hmac-Sha256', base64_encode('some_signature'));
    $request->headers->set('Content-Type', 'application/json');

    // Create webhook controller instance.
    $config_factory = $this->container->get('config.factory');
    $http_client = $this->container->get('http_client');
    $logger = $this->container->get('logger.channel.recurly_commerce_api');

    $recurly_api = new RecurlyCommerceApi($config_factory, $http_client, $logger, $key_repository);
    $controller = new \Drupal\recurly_commerce_api\Controller\RecurlyCommerceApiWebhook($recurly_api);

    // Handle the webhook.
    $response = $controller->handleIncomingWebhook($request);

    // Assert response is forbidden.
    $this->assertEquals(403, $response->getStatusCode());
  }

}
