<?php

declare(strict_types=1);

namespace Drupal\recurring_events_ical\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\recurring_events\EventInterface;
use Drupal\recurring_events_ical\EventICalInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Route controller for exporting event data in iCalendar format.
 */
class EventExportController extends ControllerBase {

  use AutowireTrait;

  public function __construct(
    protected readonly EventICalInterface $eventICal,
  ) {}

  /**
   * Returns an iCalendar response for an event series.
   *
   * @param \Drupal\recurring_events\EventInterface $eventseries
   *   The event series.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   An iCalendar file download.
   */
  public function series(EventInterface $eventseries): Response {
    return $this->response($eventseries);
  }

  /**
   * Returns an iCalendar response for an event instance.
   *
   * @param \Drupal\recurring_events\EventInterface $eventinstance
   *   The event instance.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   An iCalendar file download.
   */
  public function instance(EventInterface $eventinstance): Response {
    return $this->response($eventinstance);
  }

  /**
   * Returns an event's iCalendar data as an HTTP response.
   *
   * @param \Drupal\recurring_events\EventInterface $event
   *   An event.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   An iCalendar file download.
   */
  protected function response(EventInterface $event): Response {
    $headers = [
      'Content-Type' => 'text/calendar',
      'Content-Disposition' => 'attachment; filename="event.ics"',
    ];
    return new Response($this->eventICal->render($event), 200, $headers);
  }

}
