<?php

declare(strict_types=1);

namespace Drupal\recurring_events_registration;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\recurring_events\EventInterface;
use Symfony\Component\Routing\Route;

/**
 * AccessHandler class definition.
 */
class AccessHandler {

  use AutowireTrait;

  public function __construct(
    protected readonly TranslationInterface $translation,
    protected readonly RegistrationCreationService $creationService,
    protected readonly RouteMatchInterface $routeMatch,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * Access control based on whether event has registration.
   *
   * @return bool
   *   TRUE if event has registration, FALSE otherwise.
   */
  public function eventHasRegistration() {
    $has_registration = FALSE;
    $event_instance = $this->routeMatch->getParameter('eventinstance');
    if (!empty($event_instance)) {

      if (!$event_instance instanceof EventInterface && is_numeric($event_instance)) {
        $event_instance = $this->entityTypeManager->getStorage('eventinstance')->load($event_instance);
      }

      if ($event_instance instanceof EventInterface) {
        $this->creationService->setEventInstance($event_instance);
        $has_registration = $this->creationService->hasRegistration();
      }
    }
    return $has_registration;
  }

  /**
   * Access control based on whether the account has the right permission.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The current route.
   *
   * @return bool
   *   TRUE if user has access, FALSE otherwise.
   */
  public function userHasPermission(AccountInterface $account) {
    return $account->hasPermission('access registrant overview');
  }

  /**
   * Access control for the Event Registration List view.
   */
  public function eventRegistrationListAccess(AccountInterface $account, Route $route) {
    if (!$this->eventHasRegistration()) {
      return AccessResult::forbidden();
    }

    if (!$this->userHasPermission($account)) {
      return AccessResult::forbidden();
    }
    return AccessResult::allowed();
  }

}
