<?php

declare(strict_types=1);

namespace Drupal\recurring_events_registration\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\recurring_events\Entity\EventInstance;
use Drupal\recurring_events_registration\Entity\RegistrantInterface;

/**
 * The RegistrantController class.
 */
class RegistrantController extends ControllerBase implements ContainerInjectionInterface {

  use AutowireTrait;

  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    AccountProxyInterface $currentUser,
    protected readonly RendererInterface $renderer,
  ) {}

  /**
   * Check if registration is enabled.
   *
   * @param \Drupal\recurring_events\Entity\EventInstance $eventinstance
   *   The eventinstance entity.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   Whether access is allowed based on whether registration is enabled.
   */
  public static function hasRegistration(EventInstance $eventinstance) {
    if (!empty($eventinstance)) {
      // Static function, so we need to request the service statically, not
      // through dependency injection.
      $service = \Drupal::service('recurring_events_registration.creation_service');
      $service->setEventInstance($eventinstance);
      if ($service->hasRegistration()) {
        return AccessResult::allowed();
      }
      return AccessResult::forbidden();
    }
    return AccessResult::neutral();
  }

  /**
   * Check if the user can contact the registrants.
   *
   * @param \Drupal\recurring_events\Entity\EventInstance $eventinstance
   *   The eventinstance entity.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   Whether access is allowed based on whether registration is enabled.
   */
  public function canContactRegistrants(EventInstance $eventinstance) {
    if (!empty($eventinstance)) {
      $account = $this->entityTypeManager()->getStorage('user')->load($this->currentUser()->id());
      return AccessResult::allowedIfHasPermission($account, 'contact registrants');
    }
    return AccessResult::forbidden();
  }

  /**
   * Return a dynamic page title for a Registrant.
   *
   * @param \Drupal\recurring_events_registration\Entity\RegistrantInterface $registrant
   *   The entity for which to generate a page title.
   *
   * @return string
   *   The page title.
   */
  public function getTitle(RegistrantInterface $registrant) {
    return $registrant->label();
  }

}
